/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.integration.jms;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.jms.ConnectionFactory;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.jbpm.JbpmConfiguration;
import org.jbpm.bpel.BpelException;
import org.jbpm.bpel.integration.IntegrationService;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.svc.Service;
import org.jbpm.svc.ServiceFactory;

/**
 * @author Alejandro Guzar
 * @version $Revision: 1.10 $ $Date: 2007/07/05 09:36:35 $
 */
public class JmsIntegrationServiceFactory implements ServiceFactory {

  // injected objects, see jbpm.cfg.xml
  private String defaultConnectionFactoryName;
  private String defaultDestinationName;
  private String replyDestinationName;
  private JbpmConfiguration jbpmConfiguration;

  private ConnectionFactory defaultConnectionFactory;
  private Destination defaultDestination;
  private Destination replyDestination;

  private Map integrationControls = new HashMap();

  private static final Log log = LogFactory.getLog(JmsIntegrationServiceFactory.class);
  private static final long serialVersionUID = 1L;

  public Service openService() {
    return new JmsIntegrationService(this);
  }

  public void close() {
    Iterator integrationControlIt = integrationControls.values().iterator();
    while (integrationControlIt.hasNext()) {
      IntegrationControl integrationControl = (IntegrationControl) integrationControlIt.next();
      try {
        integrationControl.disableInboundMessageActivities();
      }
      catch (JMSException e) {
        log.warn("could not close integration control", e);
      }
    }
  }

  public JbpmConfiguration getJbpmConfiguration() {
    return jbpmConfiguration;
  }

  public IntegrationControl getIntegrationControl(ProcessDefinition processDefinition) {
    Long processId = new Long(processDefinition.getId());
    IntegrationControl integrationControl = (IntegrationControl) integrationControls.get(processId);
    if (integrationControl == null) {
      log.debug("creating integration control: processDefinition=" + processDefinition);
      integrationControl = new IntegrationControl(this);
      integrationControls.put(processId, integrationControl);
    }
    return integrationControl;
  }

  public ConnectionFactory getDefaultConnectionFactory() {
    if (defaultConnectionFactory == null && defaultConnectionFactoryName != null) {
      try {
        defaultConnectionFactory = (ConnectionFactory) lookup(defaultConnectionFactoryName);
      }
      catch (NamingException e) {
        log.debug("could not retrieve default connection", e);
      }
    }
    return defaultConnectionFactory;
  }

  public Destination getDefaultDestination() {
    if (defaultDestination == null && defaultDestinationName != null) {
      try {
        defaultDestination = (Destination) lookup(defaultDestinationName);
      }
      catch (NamingException e) {
        log.debug("could not retrieve default destination", e);
      }
    }
    return defaultDestination;
  }

  public Destination getReplyDestination() {
    if (replyDestination == null) {
      if (replyDestinationName == null) {
        throw new IllegalStateException(
            "reply destination name not specified in jbpm configuration");
      }
      try {
        replyDestination = (Destination) lookup(replyDestinationName);
      }
      catch (NamingException e) {
        throw new BpelException("could not retrieve reply destination", e);
      }
    }
    return replyDestination;
  }

  private static Object lookup(String name) throws NamingException {
    InitialContext initialContext = new InitialContext();
    try {
      return initialContext.lookup(name);
    }
    finally {
      initialContext.close();
    }
  }

  public static JmsIntegrationServiceFactory getInstance(JbpmConfiguration jbpmConfiguration) {
    return (JmsIntegrationServiceFactory) jbpmConfiguration.getServiceFactory(IntegrationService.SERVICE_NAME);
  }
}
