/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.graph.struct;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.enums.Enum;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Session;

import org.jbpm.JbpmContext;
import org.jbpm.bpel.graph.def.Activity;
import org.jbpm.bpel.graph.def.BpelVisitor;
import org.jbpm.bpel.graph.def.CompositeActivity;
import org.jbpm.bpel.sublang.def.Expression;
import org.jbpm.bpel.xml.util.DatatypeUtil;
import org.jbpm.graph.def.Transition;
import org.jbpm.graph.exe.ExecutionContext;
import org.jbpm.graph.exe.Token;

/**
 * Defines an activity to be repeated as long as the specified {@link #getCondition() condition} is
 * true.
 * @author Juan Cant
 * @version $Revision: 1.11 $ $Date: 2007/07/26 00:39:10 $
 */
public class While extends StructuredActivity {

  private static final long serialVersionUID = 1L;

  private Expression condition;
  private Loop loop = new Loop(this);

  public While() {
  }

  public While(String name) {
    super(name);
  }

  // children management
  // /////////////////////////////////////////////////////////////////////////////

  protected void addActivity(Activity activity) {
    if (!activities.isEmpty())
      removeActivity((Activity) activities.get(0));

    super.addActivity(activity);
  }

  protected void addImplicitTransitions(Activity activity) {
    if (!getEvaluateFirst())
      begin.connect(activity);

    loop.connect(activity);
    activity.connect(loop);
  }

  protected void removeImplicitTransitions(Activity activity) {
    if (!getEvaluateFirst())
      begin.disconnect(activity);

    loop.disconnect(activity);
    activity.disconnect(loop);
  }

  // while properties
  // /////////////////////////////////////////////////////////////////////////////

  public void setName(String name) {
    super.setName(name);
    loop.setName(name);
  }

  public boolean getEvaluateFirst() {
    return begin.isConnected(loop);
  }

  public void setEvaluateFirst(boolean evaluateFirst) {
    if (evaluateFirst == getEvaluateFirst())
      return;

    if (evaluateFirst)
      begin.connect(loop);
    else
      begin.disconnect(loop);
  }

  public Expression getCondition() {
    return condition;
  }

  public void setCondition(Expression condition) {
    this.condition = condition;
  }

  public Loop getLoop() {
    return loop;
  }

  public void accept(BpelVisitor visitor) {
    visitor.visit(this);
  }

  /**
   * Implements the loop behavior for the while activity
   */
  public static class Loop extends Activity {

    private static ThreadLocal entranceLocal = new ThreadLocal() {

      protected Object initialValue() {
        return new HashMap();
      }
    };

    private static final long serialVersionUID = 1L;

    private static class EntranceMark extends Enum {

      static final EntranceMark FIRST_ENTRANCE = new EntranceMark("firstEntrance");
      static final EntranceMark REENTRANCE = new EntranceMark("reentrance");

      private static final long serialVersionUID = 1L;

      private EntranceMark(String name) {
        super(name);
      }
    }

    private Log log = LogFactory.getLog(While.class);

    Loop() {
    }

    Loop(While _while) {
      super("loop");
      _while.adoptActivity(this);

      _while.getBegin().connect(this);
      connect(_while.getEnd());
    }

    public While getWhile() {
      CompositeActivity composite = getCompositeActivity();

      if (composite == null)
        return null;

      if (composite instanceof While)
        return (While) composite;

      // reacquire proxy of the proper type
      Session hbSession = JbpmContext.getCurrentJbpmContext().getSession();
      While _while = (While) hbSession.load(While.class, new Long(composite.getId()));

      // update composite activity reference
      _while.adoptActivity(this);

      return _while;
    }

    public void execute(ExecutionContext exeContext) {
      if (getMark() == EntranceMark.FIRST_ENTRANCE) {
        setMark(EntranceMark.REENTRANCE);
        log.debug("reentrance detected on "
            + getCompositeActivity()
            + " for "
            + exeContext.getToken());
        return;
      }

      While _while = getWhile();
      Expression condition = _while.getCondition();
      Token token = exeContext.getToken();

      for (;;) {
        // evaluate condition
        log.debug("evaluating " + condition + " for " + token);
        if (DatatypeUtil.toBoolean(condition.getEvaluator().evaluate(token))) {
          // mark first entrance
          setMark(EntranceMark.FIRST_ENTRANCE);

          // leave over loop transition
          leave(exeContext, (Transition) getLeavingTransitions().get(1));

          // verify entrance mark
          if (removeMark() == EntranceMark.FIRST_ENTRANCE) {
            log.debug("wait state reached from " + _while + " for " + token);
            break;
          }
          log.debug("continue " + _while + " for " + token);
        }
        else {
          // leave over end transition
          leave(exeContext, (Transition) getLeavingTransitions().get(0));
          log.debug("break " + _while + " for " + token);
          break;
        }
      }
    }

    private EntranceMark getMark() {
      return (EntranceMark) getEntranceMarks().get(this);
    }

    private void setMark(EntranceMark mark) {
      getEntranceMarks().put(this, mark);
    }

    private EntranceMark removeMark() {
      EntranceMark mark = (EntranceMark) getEntranceMarks().remove(this);
      assert mark != null : "entrance mark not found";
      return mark;
    }

    private static Map getEntranceMarks() {
      return (Map) entranceLocal.get();
    }
  }
}