/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the JBPM BPEL PUBLIC LICENSE AGREEMENT as
 * published by JBoss Inc.; either version 1.0 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
package org.jbpm.bpel.graph.struct;

import org.jbpm.bpel.graph.def.Activity;
import org.jbpm.bpel.graph.def.BpelVisitor;
import org.jbpm.graph.def.Transition;

/**
 * Defines a collection of activities to be performed sequentially in lexical order.
 * @author Juan Cant
 * @version $Revision: 1.7 $ $Date: 2007/07/26 00:39:10 $
 */
public class Sequence extends StructuredActivity {

  private static final long serialVersionUID = 1L;

  public Sequence() {
    super();
  }

  public Sequence(String name) {
    super(name);
  }

  protected boolean isChildInitial(Activity child) {
    return child.equals(getNodes().get(0));
  }

  // children management
  // /////////////////////////////////////////////////////////////////////////////

  protected void addImplicitTransitions(Activity activity) {
    if (activities.isEmpty()) {
      // connect the sequence start to the new one
      begin.connect(activity);
    }
    else {
      Activity lastActivity = (Activity) activities.get(activities.size() - 1);
      // disconnect the last activity from the sequence end
      lastActivity.disconnect(end);
      // connect the last activity to the new one
      lastActivity.connect(activity);
    }
    // connect the new activity with the sequence end
    activity.connect(end);
  }

  protected void removeImplicitTransitions(Activity activity) {
    Transition leaving = activity.getDefaultLeavingTransition();
    Transition arriving = activity.getDefaultArrivingTransition();

    activity.removeArrivingTransition(arriving);
    activity.removeLeavingTransition(leaving);

    Activity successor = (Activity) leaving.getTo();
    successor.removeArrivingTransition(leaving);
    successor.addArrivingTransition(arriving);
  }

  public void reorderNode(int oldIndex, int newIndex) {
    // remove activity from its old position and remove its incoming and outgoing connections
    Activity reorderedObject = (Activity) activities.remove(oldIndex);
    removeImplicitTransitions(reorderedObject);

    // add activity to new position and connect to its new predecessors and successors
    Activity predecessor = (newIndex == 0 ? begin : (Activity) activities.get(newIndex - 1));
    Activity successor = newIndex == activities.size() ? end : (Activity) activities.get(newIndex);

    Transition leaving = predecessor.getDefaultLeavingTransition();
    successor.removeArrivingTransition(leaving);
    reorderedObject.addArrivingTransition(leaving);
    reorderedObject.connect(successor);

    activities.add(newIndex, reorderedObject);
  }

  public void accept(BpelVisitor visitor) {
    visitor.visit(this);
  }
}