/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.util;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.regex.Pattern;

import junit.framework.TestCase;

import org.apache.log4j.Logger;

/**
 * File utilities.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class FileUtils {
	
	private static Logger logger = Logger.getLogger(FileUtils.class);

	/**
	 * Get the QA environment "build" directory file.
	 * @return QA "build" dir file.
	 */
	public static File getEnvBuildDir() {
		return new File("./build");
	}
	
	/**
	 * Creates a directory inside the build dir.
	 * <p/>
	 * If the directory already exists, it gets recreated.
	 * @param name Directory name.
	 * @return The newly created/recreated directory.
	 */
	public static File createTestDir(String name) {
		File dir = new File(getEnvBuildDir(), name);
		
		if(dir.exists()) {
			dir.delete();
		}
		dir.mkdirs();
		
		return dir;
	}
	
	/**
	 * Write the supplied data to the specified file.
	 * @param file File to write to.
	 * @param data Data to be written.
	 */
	public static void writeDataToFile(File file, byte[] data) {
		FileOutputStream fileStream = null;
		
		try {
			fileStream = new FileOutputStream(file);
			fileStream.write(data);
		} catch (IOException e) {
			String errorMsg = "Error writing to file: " + file.getAbsolutePath();
			logger.error(errorMsg, e);
			TestCase.fail(errorMsg);
		} finally {
			if(fileStream != null) {
				try {
					fileStream.flush();
					fileStream.close();
				} catch (IOException e) {
					TestCaseUtils.logAndFail("Error flushing/closing file: " + file.getAbsolutePath(), logger, new Exception());
				}
			}
		}		
	}
	
	/**
	 * Assert that the specified file exists.
	 * @param file The file in question.
	 * @param maxWait The maximum length of time (ms) to wait for for the file to appear.
	 */
	public static void assertFileExists(File file, long maxWait) {
		long endTime = System.currentTimeMillis() + maxWait;
		
		while(System.currentTimeMillis() < endTime) {
			if(file.exists()) {
				return;
			}
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
				logger.error("Thread interupt...", e);
			}
		}
		TestCaseUtils.logAndFail("File [" + file.getAbsolutePath() + "] doesn't exist.  Waited for " + maxWait + "ms.", logger, new Exception());
	}

	/**
	 * Assert that files matching the specified pattern appear in the specified directory.
	 * @param dir Directory to check.
	 * @param pattern Regexp pattern used to perform the matching.
	 * @param maxWait The maximum length of time (ms) to wait for for the files to appear.
	 */
	public static void asserFilesExist(final File dir, final String pattern, int maxWait) {
		if(!dir.isDirectory()) {
			throw new IllegalArgumentException(dir.getAbsolutePath() + " is not a directory.");
		}
		
		long endTime = System.currentTimeMillis() + maxWait;
		
		while(System.currentTimeMillis() < endTime) {
			if(hasFiles(dir, pattern)) {
				return;
			}
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
				logger.error("Thread interupt...", e);
			}
		}
		TestCaseUtils.logAndFail("Files matching the pattern [" + pattern + "] not found in directory [" 
				+ dir.getAbsolutePath() + "].  Waited for " + maxWait + "ms.", logger, new Exception());
	}
	
	/**
	 * Check for files matching the specified pattern appear in the specified directory.
	 * @param dir Directory to check.
	 * @param pattern Regexp pattern used to perform the matching.
	 * @return True if files matching that pattern exist, otherwise false.
	 */
	public static boolean hasFiles(final File dir, final String pattern) {
		FileFilter fileFilter = new FileFilter() {
			public boolean accept(File file) {
				if(!file.getParentFile().equals(dir)) {
					return false;
				}
				return Pattern.matches(pattern, file.getName());
			}
		};
		
		return dir.listFiles(fileFilter).length > 0;
	}

	/**
	 * Assert that the specified file doesn't exists.
	 * @param file The file in question.
	 * @param maxWait The maximum length of time (ms) to wait for for the file to disappear.
	 */
	public static void assertFileDoesntExists(File file, long maxWait) {
		long endTime = System.currentTimeMillis() + maxWait;
		
		while(System.currentTimeMillis() < endTime) {
			if(!file.exists()) {
				return;
			}
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
				logger.error("Thread interupt...", e);
			}
		}
		TestCaseUtils.logAndFail("File [" + file.getAbsolutePath() + "] exists.  Waited for " + maxWait + "ms.", logger, new Exception());
	}

	/**
	 * Assert that the file/dir can be deleted.
	 * <p/>
	 * Just deletes the file and makes sure the delete worked.  This ensures that nothing has an open
	 * handle on the file i.e. that code didn't "forget" to close it.  This method will fail the assertion
	 * if the target file is a non-empty directory.
	 * 
	 * @param file The file to be deleted.
	 * @param maxWait The maximum length of time (ms) to wait for for the file to disappear after calling the delete method.
	 */
	public static void assertCanDelete(File file, long maxWait) {
		if(!file.exists()) {
			TestCaseUtils.logAndFail("File [" + file.getAbsolutePath() + "] can't be deleted - it doesn't exists.", logger, null);
		}
		file.delete();
		long endTime = System.currentTimeMillis() + maxWait;
		
		if(file.isDirectory() && hasFiles(file, ".*")) {
			TestCaseUtils.logAndFail("File [" + file.getAbsolutePath() + "] can't be deleted - it's a directory and isn't empty.", logger, null);
		}
		
		while(System.currentTimeMillis() < endTime) {
			file.delete();
			if(!file.exists()) {
				return;
			}
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
				logger.error("Thread interupt...", e);
			}
		}
		TestCaseUtils.logAndFail("File [" + file.getAbsolutePath() + "] can't be deleted - a handle on it (or one of its children) may be open.  Waited " + maxWait + "ms", logger, new Exception());
	}
	
	/**
     * Read the file into a String.
     * @param file - the file to be read
     * @return String with the content of the file
     * @throws IOException - when we can't read the file
     */
    public static String readTextFile(File file) throws IOException 
    {
        StringBuffer sb = new StringBuffer(1024);
        BufferedReader reader = new BufferedReader(new FileReader(file.getPath()));
        char[] chars = new char[1];
        while( (reader.read(chars)) > -1){
            sb.append(String.valueOf(chars)); 
            chars = new char[1];
        }
        reader.close();
        return sb.toString();
    }
}
