/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.admin.console.transform;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.jboss.soa.esb.admin.console.exchange.MessageExchange;
import org.milyn.Smooks;
import org.milyn.SmooksUtil;
import org.milyn.cdr.SmooksResourceConfiguration;
import org.milyn.container.standalone.StandaloneExecutionContext;
import org.milyn.profile.DefaultProfileSet;
import org.milyn.profile.ProfileStore;

/**
 * <a href="http://milyn.codehaus.org/Smooks">Milyn Smooks</a> utility methods
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SmooksUtils {

	/**
	 * {@link SmooksResourceConfiguration} doesn't have a config "name" attribute as does the
	 * {@link TransformationResource} class.  For import and export of configs, we add and get this
	 * as a resource parameter.
	 */
	public static final String RESOURCE_NAME_PARAM = "RESERVED:RESOURCE_NAME_PARAM";

	/**
	 * Get the ordered set of message {@link TransformationResource}s targetd at the supplied message exchange.
	 * <p/>
	 * The resource are grouped by selector and ordered according to their specificity.  See 
	 * <a href="http://milyn.codehaus.org/Smooks">Milyn Smooks</a>.
	 * @param messageExchange The message exchange whose resources we're interested in.
	 * @param allResourceConfigs All the configured transformation resources on this ESB instance.
	 * @return List of transformation resources targeted at the supplied message exchange.
	 */
	public static List<TransformationResource> getMessageExchangeConfigs(MessageExchange messageExchange, List<TransformationResource> allResourceConfigs) {
		Smooks smooks = new Smooks();
		String messageProfileString = "JBossESBMessage"; // Doesn't really matter what this string is because of how we're using it here!
		StandaloneExecutionContext execContext;
		Map messageExchangeConfigsMap;
		List<TransformationResource> messageExchangeConfigsList = new ArrayList<TransformationResource>();;
		
		// Register all the configurations on the context...
		for(TransformationResource resourceConfig : allResourceConfigs) {
			SmooksUtil.registerResource(DBLinkedSmooksResourceConfiguration.getInstance(resourceConfig), smooks);
		}
		
		// Capture the message exchange properties as useragent profiles...
		String[] profilesArray = getMessageExchangeProfiles(messageExchange);

		// Create and register a message useragent with the message exchange profiles ...
		SmooksUtil.registerProfileSet(DefaultProfileSet.create( messageProfileString, profilesArray ), smooks);
    	ProfileStore profileStore = smooks.getApplicationContext().getProfileStore();
    	profileStore.getProfileSet(messageProfileString);
		execContext = smooks.createExecutionContext(messageProfileString);
		
		// Filter out the configurations for the message session...
		messageExchangeConfigsMap = execContext.getDeliveryConfig().getSmooksResourceConfigurations();
		Iterator mapIterator = messageExchangeConfigsMap.entrySet().iterator();
		while(mapIterator.hasNext()) {
			Entry mapEntry = (Entry)mapIterator.next();
			List selectorResourceList = (List)mapEntry.getValue();
			
			for(Object resourceConfig : selectorResourceList) {
				if(resourceConfig instanceof DBLinkedSmooksResourceConfiguration) {
					messageExchangeConfigsList.add(((DBLinkedSmooksResourceConfiguration)resourceConfig).persistedConfigForm);
				}
			}
		}
		

		return messageExchangeConfigsList;
	}

	/**
	 * Get the Smooks "profiles" that correspond to the supplied message exchange. 
	 * @param messageExchange The message exchange.
	 * @return List of profiles.
	 */
	private static String[] getMessageExchangeProfiles(MessageExchange messageExchange) {
		String[] profilesArray;
		List<String> profiles = new ArrayList<String>();
		
		if(messageExchange.getFromMessageType() != null) {
			profiles.add(MessageExchange.FROM_TYPE + messageExchange.getFromMessageType());
		}
		if(messageExchange.getFrom() != null) {
			profiles.add(MessageExchange.FROM + messageExchange.getFrom());
		}
		if(messageExchange.getTo() != null) {
			profiles.add(MessageExchange.TO + messageExchange.getTo());
		}
		if(messageExchange.getToMessageType() != null) {
			profiles.add(MessageExchange.TO_TYPE + messageExchange.getToMessageType());
		}

		if(profiles.isEmpty()) {
			profiles.add("$NO-PROFILES$");
		}
		
		profilesArray = new String[profiles.size()];
		profiles.toArray(profilesArray);
		
		return profilesArray;
	}

	/**
	 * Linking of the SmooksResourceConfiguration to it's persisted JBoss ESB TransformationResource
	 * form.
	 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
	 */
	private static class DBLinkedSmooksResourceConfiguration extends SmooksResourceConfiguration {
		
		private TransformationResource persistedConfigForm;
		
		public DBLinkedSmooksResourceConfiguration(String selector, String namespaceURI, String useragents, String path) {
			super(selector, namespaceURI, useragents, path);
		}
		
		private static DBLinkedSmooksResourceConfiguration getInstance(TransformationResource resourceConfig) {
			DBLinkedSmooksResourceConfiguration instance;
			
			instance = new DBLinkedSmooksResourceConfiguration(
					resourceConfig.getSelector(), resourceConfig.getNamespace(),
					resourceConfig.getUseragent(), resourceConfig.getPath());
			instance.persistedConfigForm = resourceConfig;
			
			// If there are parameters, set them too...
			if(resourceConfig.getParameters() != null) {
				for(TransformationResourceParameter param : resourceConfig.getParameters()) {
					instance.setParameter(param.getName(), param.getType(), new String(param.getValue()));
				}
			}
			
			return instance;
		}
	}
}
