/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.admin.console.exchange;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Out;
import org.jboss.soa.esb.admin.console.contract.MessageContract;
import org.jboss.soa.esb.admin.console.exchange.participant.Participant;

/**
 * Message Exchange.
 * <p/>
 * This class defines properties used to target message transformation resources at a message exchange,
 * or a subset there of.
 * <p/>
 * This class will likely be expanded to include support for other message profile types. 
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
@Name("messageFLow")
public class MessageExchange implements Serializable {
	
	private static final long serialVersionUID = 1L;
	
	public static final String FROM = "from:";
	public static final String FROM_TYPE = "from-type:";
	public static final String TO = "to:";
	public static final String TO_TYPE = "to-type:";

	@In(required=false) @Out(required=false)
	private Participant from;

	@In(required=false) @Out(required=false)
	private MessageContract fromMessageType;

	@In(required=false) @Out(required=false)
	private Participant to;

	@In(required=false) @Out(required=false)
	private MessageContract toMessageType;

	/**
	 * Set the message source {@link Participant}.
	 * @param from The from to set.
	 */
	public void setFrom(Participant from) {
		this.from = from;
	}

	/**
	 * Get the message source {@link Participant}.
	 * @return Returns the from.
	 */
	public Participant getFrom() {
		return from;
	}

	/**
	 * Set the message type associated with the exchange.
	 * @param fromMessageType The fromMessageType to set.
	 */
	public void setFromMessageType(MessageContract messageType) {
		this.fromMessageType = messageType;
	}

	/**
	 * Get the message type associated with the exchange.
	 * @return Returns the fromMessageType.
	 */
	public MessageContract getFromMessageType() {
		return fromMessageType;
	}

	/**
	 * Set the message destination {@link Participant}.
	 * @param to The to to set.
	 */
	public void setTo(Participant to) {
		this.to = to;
	}

	/**
	 * Get the message destination {@link Participant}.
	 * @return Returns the to.
	 */
	public Participant getTo() {
		return to;
	}

	/**
	 * @param toMessageType The toMessageType to set.
	 */
	public void setToMessageType(MessageContract toMessageType) {
		this.toMessageType = toMessageType;
	}

	/**
	 * @return Returns the toMessageType.
	 */
	public MessageContract getToMessageType() {
		return toMessageType;
	}

	@Out
	public String getExchangeUseragentString() {
		if(fromMessageType == null && from == null && toMessageType == null && to == null) {
			return "*";
		}

		StringBuffer stringBuffer = new StringBuffer();
		
		if(fromMessageType != null) {
			stringBuffer.append(FROM_TYPE);
			stringBuffer.append(fromMessageType != null?fromMessageType:"*");
			stringBuffer.append(from != null || toMessageType != null || to != null?" AND ":"");
		}

		if(from != null) {
			stringBuffer.append(FROM);
			stringBuffer.append(from != null?from:"*");
			stringBuffer.append(toMessageType != null || to != null?" AND ":"");
		}
		
		if(toMessageType != null) {
			stringBuffer.append(TO_TYPE);
			stringBuffer.append(toMessageType != null?toMessageType:"*");
			stringBuffer.append(to != null?" AND ":"");
		}

		if(to != null) {
			stringBuffer.append(TO);
			stringBuffer.append(to != null?to:"*");
		}
		
		return stringBuffer.toString().trim();
	}
	
	@Out
	public String getExchangeUseragentStringAsHtml() {
		if(fromMessageType == null && from == null && toMessageType == null && to == null) {
			return "All Messages";
		}

		StringBuffer stringBuffer = new StringBuffer();
		
		stringBuffer.append("<b>").append(FROM_TYPE).append("</b>");
		stringBuffer.append(fromMessageType != null?fromMessageType:"*");
		stringBuffer.append("<br/>AND<br/>");

		stringBuffer.append("<b>").append(FROM).append("</b>");
		stringBuffer.append(from != null?from:"*");
		stringBuffer.append("<br/>AND<br/>");
		
		stringBuffer.append("<b>").append(TO_TYPE).append("</b>");
		stringBuffer.append(toMessageType != null?toMessageType:"*");
		stringBuffer.append("<br/>AND<br/>");

		stringBuffer.append("<b>").append(TO).append("</b>");
		stringBuffer.append(to != null?to:"*");
		
		return stringBuffer.toString();
	}
	
	@Out
	public String getExchangeAsHtml() {
		if(fromMessageType == null && from == null && toMessageType == null && to == null) {
			return "All Messages";
		}

		StringBuffer stringBuffer = new StringBuffer();
		
		stringBuffer.append("<b>").append(FROM_TYPE).append("</b> ");
		stringBuffer.append(fromMessageType != null?fromMessageType:"*");
		stringBuffer.append("<br/>");

		stringBuffer.append("<b>").append(FROM).append("</b> ");
		stringBuffer.append(from != null?from:"*");
		stringBuffer.append("<br/>");
		
		stringBuffer.append("<b>").append(TO_TYPE).append("</b> ");
		stringBuffer.append(toMessageType != null?toMessageType:"*");
		stringBuffer.append("<br/>");

		stringBuffer.append("<b>").append(TO).append("</b> ");
		stringBuffer.append(to != null?to:"*");
		
		return stringBuffer.toString();
	}

    /**
     * Get the profile list based on this message exchange.
	 * @return The list of profiles.
	 */
	public String[] getUseragentProfiles() {
		List<String> profiles = new ArrayList<String>();
		String[] profileArray;
		
		if(fromMessageType != null) {
			profiles.add(FROM_TYPE + fromMessageType);
		}
		if(from != null) {
			profiles.add(FROM + from);
		}
		if(toMessageType != null) {
			profiles.add(TO_TYPE + toMessageType);
		}
		if(to != null) {
			profiles.add(TO + to);
		}
		
		profileArray = new String[profiles.size()];
		profiles.toArray(profileArray);

		return profileArray;
	}

	/**
	 * Reset the message exchange.
	 */
	public void reset() {
		from = null;
		fromMessageType = null;
		to = null;
		toMessageType = null;
	}
}
