/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006
 */
package org.jboss.soa.esb.admin.console;

import static org.jboss.seam.ScopeType.APPLICATION;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import javax.jms.ConnectionFactory;
import javax.jms.JMSException;
import javax.jms.TextMessage;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicPublisher;
import javax.jms.TopicSession;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jboss.seam.annotations.Create;
import org.jboss.seam.annotations.Destroy;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;

/**
 * Console update notifier.
 * <p/>
 * Sends a notification message to all parties interested in console data updates.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
@Name("updateNotifier")
@Scope(APPLICATION)
public class UpdateNotifier {
	
	/**
	 * Logger.
	 */
	private static Logger logger = Logger.getLogger(UpdateNotifier.class);

	/**
	 * Properties file.
	 */
	private static final String JBOSS_ESB_CONSOLE_PROPERTIES = "/jboss-esb-console.properties";

	private TopicConnection conn = null;
	private TopicSession session = null;
	private Topic topic = null;

	/**
	 * Send a notification to all parties listening to the update notification topic.
	 */
	public String sendNotification() {
		sendNotification(true);
		
		return "home";
	}
	
	/**
	 * Send a notification to all parties listening to the update notification topic.
	 * @param retryOnFail Perform a single retry if the notification fails.
	 */
	private void sendNotification(boolean retryOnFail) {
		if(topic == null) {
			logger.warn("Unable to successfully send Configuration notification.");
			reconnectAndNotify(retryOnFail);
			return;
		}		
		
		try {
			TopicPublisher send = session.createPublisher(topic);
			TextMessage tm = session.createTextMessage("Console");
			send.publish(tm);
			
			logger.info("Configuration Update notification sent successfully.");
		} catch (JMSException e) {
			logger.error("Configuration Update notification failed.", e);
			reconnectAndNotify(retryOnFail);
		}
	}
	
	/**
	 * Connect to the notification topic.
	 */
	@Create
	public void connect() {
		Properties properties = new Properties();
		InputStream propertiesStream = getClass().getResourceAsStream(JBOSS_ESB_CONSOLE_PROPERTIES);
		InitialContext context;
		TopicConnectionFactory connectionFactory = null;
		
		if(propertiesStream == null) {
			throw new IllegalStateException("Unexpected runtime error. '" + JBOSS_ESB_CONSOLE_PROPERTIES + "' should be available on the classpath.");
		}
		
		// Load the properties...
		try {
			properties.load(propertiesStream);
		} catch (IOException e) {
			logger.error("Error reading properties stream (" + JBOSS_ESB_CONSOLE_PROPERTIES + ").", e);
			return;
		}
		
		// Create the naming context...
		try {
			context = new InitialContext(properties);
		} catch (NamingException e) {
			logger.error("InitialContext creation failure.  Properties: " + properties, e);
			return;
		}

		String connectionFactoryRuntime = properties.getProperty(ConnectionFactory.class.getName(), "ConnectionFactory");
		try {
			connectionFactory = (TopicConnectionFactory) context.lookup(connectionFactoryRuntime);
		} catch (NamingException e) {
			logger.error("JNDI lookup of JMS Connection Factory [" + connectionFactoryRuntime + "] failed.", e);
			return;
		} catch (ClassCastException e) {
			logger.error("JNDI lookup of JMS Connection Factory failed.  Class [" + connectionFactoryRuntime + "] is not an instance of [" + ConnectionFactory.class.getName() + "].", e);
			return;
		}
		
		// Create the topic connection...
		try {
			conn = connectionFactory.createTopicConnection();
		} catch (JMSException e) {
			logger.error("Failed to open JMS TopicConnection for the Transformation configuration Update Notifier. Update Notifier not enabled!", e);
			return;
		}
		
		// Lookup the topic...
		String notificationTopic = properties.getProperty("update.notification.topic", "topic/org.jboss.soa.esb.console.transformation.Update");
		try {
			topic = (Topic) context.lookup(notificationTopic);
		} catch (NamingException e) {
			logger.warn("Topic lookup failed for the Transformation configuration Update Notifier.  Topic name '" + notificationTopic + "'. Update Notifier not enabled!");
			close();
			return;
		}
		
		// Create the TopicSession...
		try {
			session = conn.createTopicSession(false, TopicSession.AUTO_ACKNOWLEDGE);
		} catch (JMSException e) {
			logger.error("TopicSession creation failed for the Transformation configuration Update Notifier.  Update Notifier not enabled!", e);
			close();
			return;
		}
		
		// Start the connection...
		try {
			conn.start();
		} catch (JMSException e) {
			logger.error("Failed to start JMS TopicConnection for the Transformation configuration Update Notifier.  Update Notifier not enabled!", e);
			close();
			return;
		}

		logger.info("Transformation Configuration Update Notifier started!");
	}

	/**
	 * Attempt a single reconnect and notify operation.
	 * @param attemptRetry Is the retry to be attempted on this call.
	 */
	private void reconnectAndNotify(boolean attemptRetry) {
		if(attemptRetry) {
			logger.info("Attempting a single reconnect and retry.");
			close();
			connect();
			sendNotification(false);
		} else {
			logger.info("Already attempted a reconnect and retry - not attempting another.  This notification event will be lost!");
		}
	}
	
	@Destroy
	public void destroy() {
		close();
	}

	/**
	 * Close out the listener and all it's resources.
	 */
	private void close() {
		try {
			if(conn != null) {
				conn.stop();
				logger.info("Update Notifier JMS TopicConnection stopped.");
			}
		} catch (Throwable e) {
			logger.error("Failed to stop Update Notifier JMS connection.", e);
			conn = null;
		}
		try {
			if(session != null) {
				session.close();
				logger.info("Update Notifier JMS TopicSession closed.");
			}
		} catch (Throwable e) {
			logger.error("Failed to close Update Notifier JMS session.", e);
		} finally {
			session = null;
		}
		try {
			if(conn != null) {
				conn.close();
				logger.info("Update Notifier JMS TopicConnection closed.");
			}
		} catch (Throwable e) {
			logger.error("Failed to close Update Notifier JMS connection.", e);
		} finally {
			conn = null;
		}
		topic = null;
	}
}
