/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.converters;

import org.jboss.internal.soa.esb.util.PropertiesHelper;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.system.ServiceMBeanSupport;
import org.jboss.system.server.ServerConfig;
import org.jboss.system.server.ServerConfigLocator;
import org.apache.log4j.Logger;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.URL;
import java.util.Properties;

/**
 * SmooksService implementation.
 * <p/>
 * Initialises the "global" Smooks instance.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SmooksService extends ServiceMBeanSupport implements SmooksServiceMBean {

    private static Logger logger = Logger.getLogger(SmooksService.class);
    private static SmooksInstanceManager globalSmooksInstanceManager;
    private static final String CONSOLE_URL = "console.url";

    private static final String SMOOKS_PROPERTY_FILE = "smooks.esb.properties";
    private static final String SMOOKS_PROPERTY_FILE_COMMENTS = "Auto generated property file, do not edit" ;

    /**
     * The smooks default property file.
     */
    private String propertyFile ;
    
    protected void startService() throws Exception {
        super.startService();
        
        if (propertyFile != null) {
            final File baseFile = new File(propertyFile) ;
            final InputStream xmlPropertyIS ;
            if (!baseFile.isAbsolute()) {
                final URL resourceURL = Thread.currentThread().getContextClassLoader().getResource(propertyFile) ;
                xmlPropertyIS = resourceURL.openStream() ;
            } else {
                xmlPropertyIS = new FileInputStream(baseFile) ;
            }
            final byte[] propertyFileContents ;
            try {
                final ByteArrayOutputStream baos = new ByteArrayOutputStream() ;
                XMLHelper.replaceSystemProperties(XMLHelper.getXMLStreamReader(xmlPropertyIS),
                    XMLHelper.getXMLStreamWriter(baos)) ;
                propertyFileContents = baos.toByteArray() ;
            } finally {
                xmlPropertyIS.close() ;
            }
            
            final ServerConfig serverConfig = ServerConfigLocator.locate() ;
            final File dataDir = serverConfig.getServerDataDir() ;
            final File smooksPropertyFile = new File(dataDir, SMOOKS_PROPERTY_FILE) ;
            
            final ByteArrayInputStream bais = new ByteArrayInputStream(propertyFileContents) ;
            final FileOutputStream smooksPropertyOS = new FileOutputStream(smooksPropertyFile) ;
            try {
                PropertiesHelper.translateXMLToText(bais, smooksPropertyOS, SMOOKS_PROPERTY_FILE_COMMENTS) ;
            } finally {
                smooksPropertyOS.close() ;
            }
            SmooksConfiguration.setDefaultConfiguration(smooksPropertyFile.getAbsolutePath()) ;
        }

        Properties smooksEsbConfig = new SmooksConfiguration();
        String resourceConfig = smooksEsbConfig.getProperty(CONSOLE_URL);

        if(resourceConfig != null) {
            globalSmooksInstanceManager = new SmooksInstanceManager("Console", resourceConfig);
            logger.info("Centralized Smooks Instance (Console Based) started.");
        } else {
            logger.info("Centralized Smooks Instance (Console Based) instance not started.  See the '" + CONSOLE_URL + "' property in '" + SmooksConfiguration.SMOOKS_ESB_PROPERTIES + "'");
        }
    }

    protected void stopService() throws Exception {
        if(globalSmooksInstanceManager != null) {
            globalSmooksInstanceManager.close();
            logger.info("Centralized Smooks Instance (Console Based) stopped.");
        }
        
        super.stopService();
    }

    public static SmooksInstanceManager getGlobalSmooksInstanceManager() {
        return globalSmooksInstanceManager;
    }
    
    /**
     * Get the property file.
     * @return The name of the property file being used.
     */
    public String getPropertyFile() {
        return propertyFile ;
    }
    
    /**
     * Set the property file.
     * @param propertyFile The name of the property file being used.
     */
    public void setPropertyFile(final String propertyFile) {
        this.propertyFile = propertyFile ;
    }
}
