/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.converters;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.apache.log4j.Logger;
import org.milyn.Smooks;
import org.milyn.SmooksException;
import org.xml.sax.SAXException;

import javax.jms.*;
import javax.naming.Context;
import javax.naming.NamingException;
import javax.naming.InitialContext;
import java.io.InputStream;
import java.io.IOException;
import java.util.Arrays;

/**
 * Smooks instance manager.
 * <p/>
 * Manages configuration updates.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
class SmooksInstanceManager {

    private static Logger logger = Logger.getLogger(SmooksInstanceManager.class);
    private static String UPDATE_TOPIC_NAME = "update.notification.topic";
    private SmooksConfiguration config = new SmooksConfiguration();
    private String configurationSourceName;
    private String configurationSourceURI;
    private Smooks smooksInstance;
    private ConfigurationUpdateListener configUpdateListener;

    protected SmooksInstanceManager(String configurationSourceName, String configurationSourceURI) throws SmooksException {
        AssertArgument.isNotNullAndNotEmpty(configurationSourceURI, "configurationSourceURI");
        if(configurationSourceName != null) {
            this.configurationSourceName = configurationSourceName;
        } else {
            // Just set the name to be the URI...
            logger.warn("No 'configurationSourceName' specified for Smooks Configuration '" + configurationSourceURI + "'.  Defaulting name to '" + configurationSourceURI + "'.");
            this.configurationSourceName = configurationSourceURI;
        }
        this.configurationSourceURI = configurationSourceURI;

        smooksInstance = createSmooksInstance();
        configUpdateListener = new ConfigurationUpdateListener();
    }

    protected Smooks getSmooksInstance() {
        return smooksInstance;
    }

    protected void close() {
        configUpdateListener.close();
    }

    private Smooks createSmooksInstance() throws SmooksException{
        // We're not synchronizing creation of (or access to) the Smooks
        // instances, so there is a possiblility that 1+ instances of
        // Smooks could be "in circulation" for brief intervals (after an
        // update notification).  This is OK and not an issue. The
        // older instance(s) will eventually get GC'd.        
        try {
            Smooks instance = new Smooks(configurationSourceURI);
            addBaseConfigs(instance);
            return instance;
        } catch (SAXException e) {
            throw new SmooksException("Invalid Smooks resource config '" + configurationSourceURI + "'.", e);
        } catch (IOException e) {
            throw new SmooksException("Error reading Smooks resource config '" + configurationSourceURI + "'.", e);
        }
    }

    private void addBaseConfigs(Smooks instance) {
        String baseConfigs = config.getProperty("smooks.base.configs", "creators.xml");
        String[] resources = baseConfigs.split(",");
        if(logger.isDebugEnabled()) {
            logger.debug("smooks-base-configs: " + Arrays.asList(resources));
        }
        for(String resource : resources) {
            String fullResPath = "/smooks-base-configs/" + resource.trim();
            try {
                InputStream smooksRes = ClassUtil.getResourceAsStream(fullResPath, getClass());
                if(smooksRes != null) {
                    instance.addConfigurations("/smooks-base-configs/", smooksRes);
                } else {
                    logger.warn("Configured Smooks resource '" + fullResPath + "' doesn't exist on the classpath.");
                }
            } catch(Throwable t) {
                logger.warn("Exception while loading Smooks resource '" + fullResPath + "'.", t);
            }
        }
    }

    /**
     * JMS Listener for receiving configuration update notifications.
     * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
     */
    private class ConfigurationUpdateListener implements MessageListener {

    	private TopicConnection conn = null;
    	private TopicSession session = null;
    	private Topic topic = null;

        /**
    	 * Private constructor.
    	 * <p/>
    	 * Connects the listener to the topic.
    	 */
    	private ConfigurationUpdateListener() {
    		try {
    			if(!connect()) {
                    logger.warn("Failed to start Smooks Configuration Update listener.  Turn on debug logging for more info.");
                }
    		} catch(Throwable t) {
				logger.error("Unexpected error while attempting to connect Transformation configuration Update Listener.  Update listener not enabled!", t);
				close();
				return;
    		}
    	}

		/**
    	 * Receive a notification message.
    	 */
		public void onMessage(javax.jms.Message message) {
            String updatedSourceName = null;

            try {
                if(message instanceof TextMessage) {
                    updatedSourceName = ((TextMessage)message).getText();
                } else if(message instanceof ObjectMessage) {
                    try {
                        updatedSourceName = (String) ((ObjectMessage)message).getObject();
                    } catch(Throwable t) {
                        logger.error("Failed to read updateSourceName from JMS ObjectMessage.  Expecting a String object.  Error: " + t.getMessage());
                    }
                }
            } catch (JMSException e) {
                logger.warn("Error reading Smooks Update Notification configuration source name from JMS message.", e);
                return;
            }

            if(updatedSourceName != null) {
                if(updatedSourceName.equals(configurationSourceName)) {
                    logger.info("Transformation configuration update notification received for configuration source '" + configurationSourceName + "'.  Resetting SmooksTransformer in order to force a configuration re-read.");
                    smooksInstance = createSmooksInstance();
                }
            } else {
                logger.info("Transformation configuration update notification received (global notification).  Resetting SmooksTransformer in order to force a configuration re-read.");
                smooksInstance = createSmooksInstance();
            }
        }

        private ConnectionFactory getJmsConnectionFactory() throws ConfigurationException {
            ConnectionFactory factory = null;
            Context context;
            String connectionFactoryRuntime = config.getProperty(ConnectionFactory.class.getName(), "ConnectionFactory");

            context = getNamingContext();
            try {
                factory = (ConnectionFactory) context.lookup(connectionFactoryRuntime);
            } catch (NamingException e) {
                throw new ConfigurationException("JNDI lookup of JMS Connection Factory [" + connectionFactoryRuntime + "] failed.", e);
            } catch (ClassCastException e) {
                throw new ConfigurationException("JNDI lookup of JMS Connection Factory failed.  Class [" + connectionFactoryRuntime + "] is not an instance of [" + ConnectionFactory.class.getName() + "].", e);
            } finally {
                if (context!=null) {
                    try {
                        context.close();
                    } catch (NamingException ne) {
                        logger.error("Failed to close Naming Context.", ne);
                    }
                }
            }

            return factory;
        }

        private Context getNamingContext() throws ConfigurationException {
            Context context;

            try {
                context = new InitialContext(config);
            } catch (NamingException e) {
                throw new ConfigurationException("Failed to load InitialContext: " + config);
            }
            if(context == null) {
                throw new ConfigurationException("Failed to Server JNDI context.  Check that '" + Context.PROVIDER_URL + "', '" + Context.INITIAL_CONTEXT_FACTORY + "', '" + Context.URL_PKG_PREFIXES + "' are correctly configured in " + SmooksConfiguration.SMOOKS_ESB_PROPERTIES + ".");
            }
            
            return context;
        }

        /**
		 * Connect to the configured topic.
		 */
    	private boolean connect() {
            String notificationTopicName = config.getProperty(UPDATE_TOPIC_NAME, "topic/org.jboss.soa.esb.transformation.Update");
    		TopicConnectionFactory connectionFactory = null;

			logger.debug("Attempting to connect Transformation Configuration Update Listener to update notification topic '" + notificationTopicName + "'.");

    		// Get the Topic ConnectionFactory...
    		try {
				connectionFactory = (TopicConnectionFactory) getJmsConnectionFactory();
			} catch (ConfigurationException e) {
				logger.debug("Lookup of the JMS ConnectionFactory failed for the Transformation configuration Update Listener. Update listener not enabled!", e);
				return false;
			} catch (ClassCastException e) {
				logger.debug("Invalid JMS ConnectionFactory config for the Transformation configuration Update Listener.  ConnectionFactory doesn't implement " + TopicConnectionFactory.class.getName() + ". Update listener not enabled!", e);
				return false;
			}

			// Create the topic connection...
			try {
				conn = connectionFactory.createTopicConnection();
			} catch (JMSException e) {
				logger.debug("Failed to open JMS TopicConnection for the Transformation configuration Update Listener. Update listener not enabled!", e);
				return false;
			}

			// Lookup the topic...
			try {
				Context context = getNamingContext();

                topic = (Topic) context.lookup(notificationTopicName);
                context.close();
			} catch (ConfigurationException e) {
				logger.debug("Topic lookup failed for the Transformation configuration Update Listener.  Topic name '" + notificationTopicName + "'.  \n\t\tThis JMS Topic may not be deployed, or this ESB instance may not be looking at the correct JMS provider (check properties '" + UPDATE_TOPIC_NAME + "' and '" + Context.PROVIDER_URL + "' in '" + SmooksConfiguration.SMOOKS_ESB_PROPERTIES + "').  \n\t\tUpdate listener not enabled!", e);
				close();
				return false;
			} catch (NamingException e) {
				logger.debug("Topic lookup failed for the Transformation configuration Update Listener.  Topic name '" + notificationTopicName + "'.  \n\tThis JMS Topic may not be deployed, or this ESB instance may not be looking at the correct JMS provider (check properties '" + UPDATE_TOPIC_NAME + "' and '" + Context.PROVIDER_URL + "' in '" + SmooksConfiguration.SMOOKS_ESB_PROPERTIES + "').  \n\tUpdate listener not enabled!");
				close();
				return false;
			}

			// Create the TopicSession...
			try {
				session = conn.createTopicSession(false, TopicSession.AUTO_ACKNOWLEDGE);
			} catch (JMSException e) {
				logger.debug("TopicSession creation failed for the Transformation configuration Update Listener.  Update listener not enabled!", e);
				close();
				return false;
			}

			// Start the connection...
			try {
				conn.start();
			} catch (JMSException e) {
				logger.debug("Failed to start JMS TopicConnection for the Transformation configuration Update Listener.  Update listener not enabled!", e);
				close();
				return false;
			}

			// Bind "this" listener to the topic...
			try {
				TopicSubscriber topicSubscriber = session.createSubscriber(topic);
				topicSubscriber.setMessageListener(this);
			} catch (JMSException e) {
				logger.debug("Failed to start JMS TopicConnection for the Transformation configuration Update Listener.  Update listener not enabled!", e);
				close();
				return false;
			}

			// Listen for exceptions on the connection...
			try {
				conn.setExceptionListener(new ExceptionListener());
			} catch (JMSException e) {
				logger.debug("Failed to attach an ExceptionListener for the Transformation configuration Update Listener.  Update listener not enabled!", e);
				close();
				return false;
			}

            logger.info("Successfully connected update notification listener to nofification topic '" + notificationTopicName + "' for Smooks configuration source '" + configurationSourceName + "' (" + configurationSourceURI + ").");

			return true;
    	}

    	@Override
		protected void finalize() throws Throwable {
    		close();
			super.finalize();
		}

		/**
    	 * Close out the listener and all it's resources.
    	 */
		private void close() {
			try {
				if(conn != null) {
					conn.stop();
                    logger.debug("Closing JMS Connection for update notification listener for Smooks configuration source '" + configurationSourceName + "'.");
				}
			} catch (Throwable e) {
				logger.error("Failed to stop Update Listener JMS connection.", e);
				conn = null;
			}
			try {
				if(session != null) {
					session.close();
                    logger.debug("Closing JMS Session for update notification listener for Smooks configuration source '" + configurationSourceName + "'.");
				}
			} catch (Throwable e) {
				logger.error("Failed to close Update Listener JMS session.", e);
			} finally {
				session = null;
			}
			try {
				if(conn != null) {
					conn.close();
                    logger.debug("Closing JMS Topic for update notification listener for Smooks configuration source '" + configurationSourceName + "'.");
				}
			} catch (Throwable e) {
				logger.error("Failed to close Update Listener JMS connection.", e);
			} finally {
				conn = null;
			}
			topic = null;
            logger.debug("Update notification listener for Smooks configuration source '" + configurationSourceName + "' is now stopped!");
		}

		/**
		 * Exception Listener.
		 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
		 */
		private class ExceptionListener implements javax.jms.ExceptionListener {

			/**
			 * We want this listener to handle only one exception.
			 * It will close all existing resources and create a new instance
			 * once it successfully reconnects.
			 */
			private boolean hasHandledOneException = false;

			/* (non-Javadoc)
			 * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
			 */
			public void onException(JMSException e) {
				synchronized (ExceptionListener.class) {
					if(!hasHandledOneException) {
                        if(!logger.isDebugEnabled()) {
                            logger.warn("JMS Exception on Transformation Configuration Update Listener: " + e.getMessage());
                        } else {
                            logger.debug("JMS Exception on Transformation Configuration Update Listener.", e);
                        }
                        close();
						while(!connect()) {
							try {
								Thread.sleep(5000);
							} catch (InterruptedException e1) {
                                if(!logger.isDebugEnabled()) {
	    							logger.warn("Interrupted during reconnect attempt.  Aborting reconnect!  Will need restart to reconnect: " + e.getMessage());
                                } else {
                                    logger.debug("Interrupted during reconnect attempt.  Aborting reconnect!  Will need restart to reconnect.", e);
                                }
                            }
						}
						hasHandledOneException = true;
					}
				}
			}
		}
    }

}
