/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.routing.cbr;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.util.List;

import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.services.routing.MessageRouterException;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;
import org.junit.BeforeClass;
import org.junit.Test;
/**
 * 
 * @author kurt.stam@redhat.com
 *
 */
public class ContentBasedRoutingUnitTest
{
	@Test
	public void routeSerializedMessage()
	{
		try {
            //new messages
			Message message = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
			//set some properties inside the Message
			message.getProperties().setProperty("prop1", "val1");
			message.getProperties().setProperty("prop2", "val2");
			//set the body inside the Message
			message.getBody().add(("TEST BODY").getBytes());
			//set some object attachments inside the Message
			message.getAttachment().addItem(new String("TEST ATTACHMENT1"));
			message.getAttachment().addItem(new String("TEST ATTACHMENT2"));
			
			ContentBasedRouter cbr = ContentBasedRouterFactory.getRouter(org.jboss.soa.esb.actions.ContentBasedRouter.DEFAULT_CBR_CLASS);
			List<String> destinationServices = cbr.route("JBossESBRules.drl",false,message,null);
			assertEquals(destinationServices.iterator().next(),"serialized-destination");
		} catch (MessageRouterException e) {
			e.printStackTrace();
			assertTrue(false);
		}
	}
	
	@Test
	public void routeXMLMessage()
	{
		try {
			//add new messages
			Message message = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
			//set some properties inside the Message
			message.getProperties().setProperty("prop1", "val1");
			message.getProperties().setProperty("prop2", "val2");
			//set the body inside the Message
			message.getBody().add(("TEST BODY").getBytes());
			//set some object attachments inside the Message
			message.getAttachment().addItem(new String("TEST ATTACHMENT1"));
			message.getAttachment().addItem(new String("TEST ATTACHMENT2"));
			
			ContentBasedRouter cbr = ContentBasedRouterFactory.getRouter(org.jboss.soa.esb.actions.ContentBasedRouter.DEFAULT_CBR_CLASS);
			List<String> destinationServices = cbr.route("JBossESBRules.drl",false,message,null);
			assertEquals(destinationServices.iterator().next(),"xml-destination");
		} catch (MessageRouterException e) {
			e.printStackTrace();
		}
	}
	
	@Test
	public void routeXMLMessageUsingXPathMatch()
	{
		try {
			//add new messages
			Message message = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
			//set some properties inside the Message
			message.getProperties().setProperty("prop1", "val1");
			message.getProperties().setProperty("prop2", "val2");
			//set the body inside the Message
			message.getBody().add(("<jbossesb>TEST BODY</jbossesb>").getBytes());
			//set some object attachments inside the Message
			message.getAttachment().addItem(new String("TEST ATTACHMENT1"));
			message.getAttachment().addItem(new String("TEST ATTACHMENT2"));
			
			ContentBasedRouter cbr = ContentBasedRouterFactory.getRouter(org.jboss.soa.esb.actions.ContentBasedRouter.DEFAULT_CBR_CLASS);
			List<String> destinationServices = cbr.route("JBossESBRules-XPath.drl","XPathLanguage.dsl",false,message,null);
			assertEquals(destinationServices.iterator().next(),"XML_XPath_Destination");
		} catch (MessageRouterException e) {
			e.printStackTrace();
		}
	}
	
	@Test
	public void routeXMLMessageUsingXPathEquals()
	{
		try {
			//add new messages
			Message message = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
			//set the body inside the Message
			message.getBody().add(("<Dave>rocks</Dave>").getBytes());
		
			ContentBasedRouter cbr = ContentBasedRouterFactory.getRouter(org.jboss.soa.esb.actions.ContentBasedRouter.DEFAULT_CBR_CLASS);
			List<String> destinationServices = cbr.route("JBossESBRules-XPath.drl","XPathLanguage.dsl",false,message,null);
			assertEquals(destinationServices.iterator().next(),"XML_XPath_Dave_Destination");
		} catch (MessageRouterException e) {
			e.printStackTrace();
		}
	}
	
	@Test
	public void routeXMLMessageUsingXPathGreaterThen()
	{
		try {
			//add new messages
			Message message = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
			//set the body inside the Message
			message.getBody().add(("<price>1.55</price>").getBytes());
			
			ContentBasedRouter cbr = ContentBasedRouterFactory.getRouter(org.jboss.soa.esb.actions.ContentBasedRouter.DEFAULT_CBR_CLASS);
			List<String> destinationServices = cbr.route("JBossESBRules-XPath.drl","XPathLanguage.dsl",false,message,null);
			assertEquals(destinationServices.iterator().next(),"XML_XPath_GreaterThan_Destination");
		} catch (MessageRouterException e) {
			e.printStackTrace();
		}
	}
	
	@Test
	public void routeXMLMessageUsingXPathLessThen()
	{
		try {
			//add new messages
			Message message = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
			//set the body inside the Message
			message.getBody().add(("<price>0.55</price>").getBytes());
			
			ContentBasedRouter cbr = ContentBasedRouterFactory.getRouter(org.jboss.soa.esb.actions.ContentBasedRouter.DEFAULT_CBR_CLASS);
			List<String> destinationServices = cbr.route("JBossESBRules-XPath.drl","XPathLanguage.dsl",false,message,null);
			assertEquals(destinationServices.iterator().next(),"XML_XPath_LessThan_Destination");
		} catch (MessageRouterException e) {
			e.printStackTrace();
		}
	}
	
	@BeforeClass
	public static void runBeforeAllTests() throws Exception 
	{
		try {
			TestEnvironmentUtil.setESBPropertiesFileToUse();
		} catch (Exception e) {
			e.printStackTrace();
			System.out.println("We should stop testing, since we don't any config properties");
			assertTrue(false);
		}
	}
	
	public static junit.framework.Test suite() {
		return new JUnit4TestAdapter(ContentBasedRoutingUnitTest.class);
	}
	
}
