/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.services.jbpm.actionhandlers;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.addressing.EPR;
import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmContext;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.graph.exe.Token;
import org.junit.Before;
import org.junit.Test;
/**
 * Tests the capabilities of jBPM which we are using to handle exceptions.
 * 
 * @author kstam
 *
 */
public class ExceptionFlowTest 
{
    private static String PROCESS_DEF_XML = "testExceptionHandling.xml";
	private static Logger logger = Logger.getLogger(ExceptionFlowTest.class);
    private static long processInstanceId;
    
    private static EPR epr1;
    private static MockCourier courier1;
    private static String MOCK_CATEGORY="MockCategory";
    private static String MOCK_SERVICE ="MockService";

    @Before
    public void setup() throws Exception
    {
        MockCourierFactory.install();
        MockRegistry.install();
        epr1 = new EPR(new URI("test1"));
        courier1 = new MockCourier(true);
        MockRegistry.register(MOCK_CATEGORY, MOCK_SERVICE, epr1, courier1);
        
        logger.info("Setting up jBPM");
        JbpmConfiguration jbpmConfiguration = JbpmConfiguration.getInstance();
        JbpmContext jbpmContext = jbpmConfiguration.createJbpmContext();
        jbpmContext.getJbpmConfiguration().getJobExecutor().start();
      
        //Extract a process definition from the processdefinition.xml file.
        ProcessDefinition processDefinition 
            = ProcessDefinition.parseXmlResource(PROCESS_DEF_XML);
        assertNotNull(processDefinition);
        //Create an instance of the process definition.
        jbpmContext.deployProcessDefinition(processDefinition);
        //processInstance = new ProcessInstance(processDefinition);
        ProcessInstance processInstance = jbpmContext.newProcessInstance("testExceptionHandling");
        assertTrue(
                "Instance is in start state", 
                "start".equals(processInstance.getRootToken().getNode().getName()));
        processInstanceId = processInstance.getId();
        jbpmContext.close();
    }

    /**
     * Tests timeout of the first node (service1). The time-out-transition will
     * be taken and we should en up in the ExceptionHandling end-state.
     * 
     * @throws Exception
     */
    @Test
    public void timeOut() throws Exception
    { 
        JbpmConfiguration jbpmConfiguration = JbpmConfiguration.getInstance();
        JbpmContext jbpmContext = jbpmConfiguration.createJbpmContext();
        ProcessInstance processInstance = jbpmContext.loadProcessInstance(processInstanceId);
        
        Token token = processInstance.getRootToken();
        assertEquals("start",token.getNode().getName());
        //Move the process to service1
        processInstance.signal();
        jbpmContext.close();
        //Our timer fires after 1000 millis, so let's wait for that
        int seconds=0;
        while(true) {
            jbpmContext = jbpmConfiguration.createJbpmContext();
            processInstance = jbpmContext.loadProcessInstance(processInstanceId);
            token = processInstance.getRootToken();
            logger.info(seconds++ + " Node=" + token.getNode().getName());
            if ("ExceptionHandling".equals(token.getNode().getName())) break; //we're done waiting
            if (seconds > 20) break; // the timer must not have fired so we are bailing (and failing)
            jbpmContext.close();
            try { Thread.sleep(1000); } catch (InterruptedException e) {e.printStackTrace();}
        }
        assertEquals("ExceptionHandling",token.getNode().getName());
    }
    
    /**
     * Tests bypassing the timout and taking the exception transition on service2.
     *
     */
    @Test
    public void takeExceptionTransition() {
        JbpmConfiguration jbpmConfiguration = JbpmConfiguration.getInstance();
        JbpmContext jbpmContext = jbpmConfiguration.createJbpmContext();
        ProcessInstance processInstance = jbpmContext.loadProcessInstance(processInstanceId);
        
        Token token = processInstance.getRootToken();
        assertEquals("start",token.getNode().getName());
        //Move the process to service1
        processInstance.signal();
        assertEquals("Service1",token.getNode().getName());
//      Move the process to service2, no timeout on service1
        processInstance.signal();
        assertEquals("Service2",token.getNode().getName());
        //Move the service2 ExceptionHandling
        processInstance.signal("exception");
        assertEquals("ExceptionHandling",token.getNode().getName());
    }
    
    /**
     * Tests bypassing the timout and setting exception of 3 to take the conditional transition.
     *
     */
    @Test
    public void takeConditionalTransition() {
        JbpmConfiguration jbpmConfiguration = JbpmConfiguration.getInstance();
        JbpmContext jbpmContext = jbpmConfiguration.createJbpmContext();
        ProcessInstance processInstance = jbpmContext.loadProcessInstance(processInstanceId);
        
        Token token = processInstance.getRootToken();
        assertEquals("start",token.getNode().getName());
        //Move the process to service1
        processInstance.signal();
        assertEquals("Service1",token.getNode().getName());
//      Move the process to service2, no timeout on service1
        processInstance.signal();
        assertEquals("Service2",token.getNode().getName());
        //Move the service3
        processInstance.signal();
        assertEquals("Service3",token.getNode().getName());
        processInstance.getContextInstance().createVariable("exceptionCode",3);
        processInstance.signal();
        //Based on the setting of exceptionCode of 3 the conditional transition
        //should be taken.
        assertEquals("ExceptionHandling",token.getNode().getName());
    }
    
    /**
     * Tests bypassing the timout and setting exception of 3 to take the conditional transition.
     *
     */
    @Test
    public void doNotTakeConditionalTransition() {
        JbpmConfiguration jbpmConfiguration = JbpmConfiguration.getInstance();
        JbpmContext jbpmContext = jbpmConfiguration.createJbpmContext();
        ProcessInstance processInstance = jbpmContext.loadProcessInstance(processInstanceId);
        
        Token token = processInstance.getRootToken();
        assertEquals("start",token.getNode().getName());
        //Move the process to service1
        processInstance.signal();
        assertEquals("Service1",token.getNode().getName());
//      Move the process to service2, no timeout on service1
        processInstance.signal();
        assertEquals("Service2",token.getNode().getName());
        //Move the service3
        processInstance.signal();
        assertEquals("Service3",token.getNode().getName());
        processInstance.getContextInstance().createVariable("exceptionCode",2);
        processInstance.signal();
        //Based on the setting of exceptionCode of 2 the conditional transition
        //should not be taken.
        assertEquals("end",token.getNode().getName());
    }
    
    /**
     * Tests bypassing the timout and setting exception of 3 to take the conditional transition.
     *
     */
    @Test
    public void doNotTakeConditionalTransition2() {
        JbpmConfiguration jbpmConfiguration = JbpmConfiguration.getInstance();
        JbpmContext jbpmContext = jbpmConfiguration.createJbpmContext();
        ProcessInstance processInstance = jbpmContext.loadProcessInstance(processInstanceId);
        
        Token token = processInstance.getRootToken();
        assertEquals("start",token.getNode().getName());
        //Move the process to service1
        processInstance.signal();
        assertEquals("Service1",token.getNode().getName());
//      Move the process to service2, no timeout on service1
        processInstance.signal();
        assertEquals("Service2",token.getNode().getName());
        //Move the service3
        processInstance.signal();
        assertEquals("Service3",token.getNode().getName());
        processInstance.signal();
        //Based on the not setting of exceptionCode the conditional transition
        //should not be taken.
        assertEquals("end",token.getNode().getName());
    }
    
    public static junit.framework.Test suite(){
        return new JUnit4TestAdapter(ExceptionFlowTest.class);
    }

}
