/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.testutils;

import java.io.IOException;
import java.io.StringReader;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Utility methods for string manipulation and testing.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class StringUtils {

	/**
	 * Compare the supplied strings, ignoring line break characters.
	 * @param string1 The 1st string.
	 * @param string2 The 2nd string.
	 * @param ignorecase True if a case insensitive comparison is to
	 * be performed, otherwise false. 
	 * @return True if the strings are equal, otherwise false.
	 */
	public static boolean equalsIgnoreLinebreaks(String string1, String string2, boolean ignorecase) {
		string1 = removeChar(string1, '\r');
		string1 = removeChar(string1, '\n');
		string2 = removeChar(string2, '\r');
		string2 = removeChar(string2, '\n');

		if(ignorecase) {
			return string1.equalsIgnoreCase(string2);
		} else {
			return string1.equals(string2);
		}
	}
	
	/**
	 * Delete all instances of the specified character from the supplied string.
	 * @param string The string to be opperated on.
	 * @param character The character to be removed.
	 */
	public static String removeChar(String string, char character) {
		StringBuffer stringBuf = new StringBuffer(string);
		int index = 0;
		
		while(index < stringBuf.length()) {
			if(stringBuf.charAt(index) == character) {
				stringBuf.deleteCharAt(index);
			} else {
				index++;
			}
		}
		
		return stringBuf.toString();
	}
	
	/**
	 * Compare the specified contents as XML.
	 * @param content1 The first content.
	 * @param content2 The second content.
	 * @return true if equals, false otherwise.
	 * @throws ParserConfigurationException
	 * @throws SAXException
	 * @throws IOException
	 */
        public static boolean compareXMLContent(final String content1, final String content2)
            throws ParserConfigurationException, SAXException, IOException
        {
            try {
                final SAXParserFactory parserFactory = SAXParserFactory.newInstance() ;
                parserFactory.setNamespaceAware(true) ;
    
                final SAXParser parser = parserFactory.newSAXParser() ;
                final IdentitySAXHandler handler1;
                final IdentitySAXHandler handler2;
    
                try {
                    handler1 = new IdentitySAXHandler() ;
                    parser.parse(new InputSource(new StringReader(content1)), handler1) ;
                } catch(SAXException e) {
                    System.out.println("Failed to parse content1 [" + content1 + "].");
                    throw e;
                }
    
                try {
                    handler2 = new IdentitySAXHandler() ;
                    parser.parse(new InputSource(new StringReader(content2)), handler2) ;
                } catch(SAXException e) {
                    System.out.println("Failed to parse content2 [" + content2 + "].");
                    throw e;
                }
    
                return (handler1.getRootElement().equals(handler2.getRootElement())) ;
            } catch(IOException e) {
                e.printStackTrace();
                throw e;
            }
        }
}
