/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.notification.jms;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.net.URI;
import java.net.URISyntaxException;

import javax.jms.JMSException;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.Before;
import org.junit.Test;
import org.mockejb.jms.TextMessageImpl;

/**
 * Unit test for DefaultJMSPropertiesSetter.
 * 
 * @author <a href="daniel.bevenius@redpill.se">Daniel Bevenius</a>				
 *
 */
public class DefaultJMSPropertiesSetterUnitTest
{
	@SuppressWarnings ("unused")
	private Logger log = Logger .getLogger( DefaultJMSPropertiesSetterUnitTest.class ); 
	
	private final String bodyContent = "hello";
	private final String messageID = "1234-junittest";
	private final String propertyKey = "testKey";
	private TextMessageImpl toJMSMessage;
	private DefaultJMSPropertiesSetter strategy;
	private Message fromESBMessage;
	
	@Test
	public void setJMSProperties_negative() throws JMSException, URISyntaxException
	{
		strategy.setJMSProperties( fromESBMessage, toJMSMessage );
		assertNull (toJMSMessage.getJMSCorrelationID() );
	}
	
	@Test
	public void setJMSProperties_with_set_correlationID() throws JMSException, URISyntaxException
	{
		fromESBMessage.getHeader().getCall().setMessageID( new URI( messageID ) ) ;
		
		strategy.setJMSProperties( fromESBMessage, toJMSMessage );
		
		assertEquals ( messageID,  toJMSMessage.getJMSCorrelationID() );
	}
	
	@Test
	public void setJMSProperties_with_custom_correlationID() throws JMSException, URISyntaxException
	{
		final String expectedCorrelationID = "testCorrelationID22333";
		final URI correlationURI = new URI( JMSEpr.JMS_PROTOCOL, "correlationID" , expectedCorrelationID  );
		fromESBMessage.getHeader().getCall().setRelatesTo( correlationURI );
		
		strategy.setJMSProperties( fromESBMessage, toJMSMessage );
		
		assertEquals ( expectedCorrelationID,  toJMSMessage.getJMSCorrelationID() );
	}
	
	@Test
	public void setJMSProperties_with_set_expiration() throws JMSException, URISyntaxException
	{
		long expire = 1883l;
		fromESBMessage.getProperties().setProperty( JMSPropertiesSetter.JMS_EXPIRATION, expire );
		
		strategy.setJMSProperties( fromESBMessage, toJMSMessage );
		
		assertEquals ( expire, toJMSMessage.getJMSExpiration() );
	}
	
	@Test
	public void setJMSProperties_with_set_properties_negative() throws JMSException, URISyntaxException
	{
		final String propertyKey = "org.jboss.soa.esb.message.time.dod";
		final String propertyValue = "myPropertyValue";
		fromESBMessage.getProperties().setProperty( propertyKey, propertyValue );
		
		strategy.setJMSProperties( fromESBMessage, toJMSMessage );
		
		String value = toJMSMessage.getStringProperty( propertyKey );
		assertTrue ( value == null  );
	}
	
	@Test
	public void setJMSProperties_with_set_String_property() throws JMSException, URISyntaxException
	{
		final String propertyValue = "myPropertyValue";
		fromESBMessage.getProperties().setProperty( propertyKey, propertyValue );
		
		strategy.setJMSProperties( fromESBMessage, toJMSMessage );
		assertEquals ( propertyValue, toJMSMessage.getStringProperty( propertyKey ) );
	}
	
	@Test
	public void setJMSProperties_with_set_boolean_property() throws JMSException, URISyntaxException
	{
		final boolean propertyValue = true;
		fromESBMessage.getProperties().setProperty( propertyKey, propertyValue );
		
		strategy.setJMSProperties( fromESBMessage, toJMSMessage );
		assertEquals ( propertyValue, toJMSMessage.getBooleanProperty( propertyKey ) );
	}
	
	@Test
	public void setJMSProperties_with_set_short_property() throws JMSException, URISyntaxException
	{
		final short propertyValue = 2;
		fromESBMessage.getProperties().setProperty( propertyKey, propertyValue );
		
		strategy.setJMSProperties( fromESBMessage, toJMSMessage );
		assertEquals ( propertyValue, toJMSMessage.getShortProperty( propertyKey ) );
	}
	
	@Test
	public void setJMSProperties_with_set_int_property() throws JMSException, URISyntaxException
	{
		final int propertyValue = 2;
		fromESBMessage.getProperties().setProperty( propertyKey, propertyValue );
		
		strategy.setJMSProperties( fromESBMessage, toJMSMessage );
		assertEquals ( propertyValue, toJMSMessage.getIntProperty( propertyKey ) );
	}
	
	@Test
	public void setJMSProperties_with_set_long_property() throws JMSException, URISyntaxException
	{
		final long propertyValue = 102l;
		fromESBMessage.getProperties().setProperty( propertyKey, propertyValue );
		
		strategy.setJMSProperties( fromESBMessage, toJMSMessage );
		assertEquals ( propertyValue, toJMSMessage.getLongProperty( propertyKey ) );
	}
	
	@Test
	public void setJMSProperties_with_set_float_property() throws JMSException, URISyntaxException
	{
		final float propertyValue = 1024;
		fromESBMessage.getProperties().setProperty( propertyKey, propertyValue );
		
		strategy.setJMSProperties( fromESBMessage, toJMSMessage );
		assertEquals ( propertyValue, toJMSMessage.getFloatProperty( propertyKey ) );
	}
	
	@Test
	public void setJMSProperties_with_set_double_property() throws JMSException, URISyntaxException
	{
		final double propertyValue = 1024d;
		fromESBMessage.getProperties().setProperty( propertyKey, propertyValue );
		
		strategy.setJMSProperties( fromESBMessage, toJMSMessage );
		assertEquals ( propertyValue, toJMSMessage.getDoubleProperty( propertyKey ) );
	}
	
	@Test
	public void setJMSProperties_with_set_byte_property() throws JMSException, URISyntaxException
	{
		final byte propertyValue = 22;
		fromESBMessage.getProperties().setProperty( propertyKey, propertyValue );
		
		strategy.setJMSProperties( fromESBMessage, toJMSMessage );
		assertEquals ( propertyValue, toJMSMessage.getByteProperty( propertyKey ) );
	}
	
	@Before
	public void before() throws JMSException
	{
		toJMSMessage = new TextMessageImpl();
		toJMSMessage.setText( bodyContent );
		fromESBMessage = MessageFactory.getInstance().getMessage();
		strategy = new DefaultJMSPropertiesSetter();
	}
	
	/*
	 * Just here to help Ant to find annotated test.
	 */
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( DefaultJMSPropertiesSetterUnitTest.class );
	}	

}
