/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.message.tests;

import junit.framework.TestCase;
import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.message.format.xml.MessageImpl;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.*;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.util.Util;

import java.net.URL;

/**
 * Unit tests for the Class class.
 * 
 * @author Mark Little
 */

public class XMLMessageUnitTest extends TestCase
{
	private Logger log = Logger.getLogger( XMLMessageUnitTest.class );
	
	public void testToXML()
	{
		// get XML message

		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);

		assertNotNull("created message", msg);

		try
		{
			final String xmlRepresentation = msgToXML((MessageImpl)msg) ;

			log.debug("Message looks like: " + xmlRepresentation);
		}
		catch (Exception ex)
		{
			fail(ex.toString());
		}
	}

	public void testFromXML()
	{
		// get XML message

		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);

		assertNotNull("created message", msg);

		try
		{
			final String xmlRepresentation = msgToXML((MessageImpl)msg) ;
			@SuppressWarnings("unused")
			final MessageImpl nImpl = msgFromXML(xmlRepresentation) ;
		}
		catch (Exception ex)
		{
			fail(ex.toString());
		}
	}

	public void testHeader ()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);

		assertNotNull("created message", msg);

		Call call = new Call();

		msg.getHeader().setCall(call);
		
		call = msg.getHeader().getCall();
		
		assertNotNull("message call", call);
		
		try
		{
			msg.getHeader().setCall(null);
			
			fail();
		}
		catch (IllegalArgumentException ex)
		{
		}
		catch (Exception ex)
		{
			fail(ex.toString());
		}
	}

	public void testInvalidAdd ()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);

		assertNotNull("created message", msg);
		
		try
		{
			msg.getBody().add(null, null);
			
			fail();
		}
		catch (IllegalArgumentException ex)
		{
		}
		catch (Exception ex)
		{
			fail(ex.toString());
		}
	}
	
	public void testAddBytes ()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);

		assertNotNull("created message", msg);
		
		String testString = "test";
		
		msg.getBody().add(BytesBody.BYTES_LOCATION, testString.getBytes());
		
		try
		{
			final String xmlRepresentation = msgToXML((MessageImpl)msg) ;
			log.debug("Document is "+xmlRepresentation);
			
			final MessageImpl nImpl = msgFromXML(xmlRepresentation) ;
			
			String val = new String((byte[]) nImpl.getBody().get(BytesBody.BYTES_LOCATION));
			
			assertEquals(val, testString);
		}
		catch (Exception ex)
		{			
			fail(ex.toString());
		}
	}
	
	public void testReplace ()
	{
		Message msg1 = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);

		assertNotNull("created message", msg1);
		
		String foo = "foo";
		
		Message msg2 = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);

		assertNotNull("second created message", msg2);
		
		String bar = "bar";
		
		msg1.getBody().add(BytesBody.BYTES_LOCATION, foo.getBytes());
		msg2.getBody().add(BytesBody.BYTES_LOCATION, bar.getBytes());
		
		msg1.getBody().replace(msg2.getBody());
		
		String foobar = new String((byte[]) msg1.getBody().get(BytesBody.BYTES_LOCATION));
		
		assertEquals(foobar, "bar");
	}
	
	public void testMerge ()
	{
		Message msg1 = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);

		assertNotNull("created message", msg1);
		
		String foo = "foo";
		
		Message msg2 = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);

		assertNotNull("second created message", msg2);
		
		String bar = "bar";
		
		msg1.getBody().add(BytesBody.BYTES_LOCATION, foo.getBytes());
		
		assertEquals(new String((byte[]) msg1.getBody().get(BytesBody.BYTES_LOCATION)), "foo");
		
		msg2.getBody().add(BytesBody.BYTES_LOCATION, bar.getBytes());
		
		msg1.getBody().merge(msg2.getBody());
		
		String foobar = new String((byte[]) msg1.getBody().get(BytesBody.BYTES_LOCATION));
		
		assertEquals(foobar, "bar");
	}
	
	public void testAddObjects ()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);

		assertNotNull("created message", msg);
		
		ExampleObject value = new ExampleObject(1234);
		
		msg.getBody().add("foo", value);
		
		try
		{
			final String xmlRepresentation = msgToXML((MessageImpl)msg) ;
			log.debug("Document is "+xmlRepresentation);
			
			final MessageImpl nImpl = msgFromXML(xmlRepresentation) ;
			
			ExampleObject foo = (ExampleObject) nImpl.getBody().get("foo");
			
			assertNotNull("deserialised example object", foo) ;
			
			assertEquals((foo.getValue() == value.getValue()), true);
		}
		catch (Exception ex)
		{
			log.error(ex);
			
			fail(ex.toString());
		}
	}
	
	public void testGetNames ()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);

		assertNotNull("created message", msg);
		
		ExampleObject value = new ExampleObject(1234);
		
		msg.getBody().add("foo", value);
		msg.getBody().add("bar", value);
		
		String[] names = msg.getBody().getNames();
		
		assertNotNull(names);
		
		assertEquals(names.length, 2);
		
		/*
		 * The array comes back ordered alphabetically.
		 */
		
		assertEquals(names[0], "bar");
		assertEquals(names[1], "foo");
		
		try
		{
			final String xmlRepresentation = msgToXML((MessageImpl)msg) ;
			log.debug("Document is "+xmlRepresentation);
			
			final MessageImpl nImpl = msgFromXML(xmlRepresentation) ;
			
			ExampleObject foo = (ExampleObject) nImpl.getBody().get("foo");
			
			assertNotNull("deserialised example object", foo) ;
			
			assertEquals((foo.getValue() == value.getValue()), true);
			
			names = nImpl.getBody().getNames();
			
			assertNotNull(names);
			
			assertEquals(names.length, 2);	
			
			/*
			 * The array comes back ordered alphabetically.
			 */
			
			assertEquals(names[0], "bar");
			assertEquals(names[1], "foo");
		}
		catch (Exception ex)
		{
			fail(ex.toString());
		}
	}
	
	public void testRemoveObjects ()
	{
		Message msg = MessageFactory.getInstance().getMessage(
				MessageType.JBOSS_XML);

		assertNotNull("created message", msg);
		
		ExampleObject value = new ExampleObject(1234);
		
		msg.getBody().add("bar", value);
		
		msg.getBody().remove("bar");
		
		try
		{
			final String xmlRepresentation = msgToXML((MessageImpl)msg) ;
			log.debug("Document is "+xmlRepresentation);
			
			final MessageImpl nImpl = msgFromXML(xmlRepresentation) ;
			
			ExampleObject foo = (ExampleObject) nImpl.getBody().get("bar");
			
			assertNull("deserialised example object", foo);
		}
		catch (Exception ex)
		{			
			fail(ex.toString());
		}	
	}
	
	public void testJmsEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);
		JMSEpr epr = new JMSEpr(JMSEpr.TOPIC_TYPE, "foo", "bar");
		
		msg.getHeader().getCall().setTo(epr);
		
		try
		{
			final String xmlRepresentation = msgToXML((MessageImpl)msg) ;
			log.debug("Document is "+xmlRepresentation);
			
			final MessageImpl nImpl = msgFromXML(xmlRepresentation) ;
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();
			
			assertEquals(theEpr instanceof JMSEpr, true);
			
			assertEquals(((JMSEpr) theEpr).getConnectionFactory(), "bar");
		}
		catch (Exception ex)
		{			
			fail(ex.toString());
		}
	}
	
	public void testHttpEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);

		try
		{
			HTTPEpr epr = new HTTPEpr("http://www.foo.bar");
			
			msg.getHeader().getCall().setTo(epr);
			
			final String xmlRepresentation = msgToXML((MessageImpl)msg) ;
			log.debug("Document is "+xmlRepresentation);
			
			final MessageImpl nImpl = msgFromXML(xmlRepresentation) ;
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();
			
			assertEquals(theEpr instanceof HTTPEpr, true);
			
			assertEquals(((HTTPEpr) theEpr).getURL().toString(), "http://www.foo.bar");
		}
		catch (Exception ex)
		{			
			fail(ex.toString());
		}
	}

	public void testEmailEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);

		try
		{
			EmailEpr epr = new EmailEpr(EmailEpr.SMTP_PROTOCOL, "foo.bar", "25", "me", "password");
			
			msg.getHeader().getCall().setTo(epr);
			
			final String xmlRepresentation = msgToXML((MessageImpl)msg) ;
			log.debug("Document is "+xmlRepresentation);
			
			final MessageImpl nImpl = msgFromXML(xmlRepresentation) ;
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();
			
			assertEquals(theEpr instanceof EmailEpr, true);
			
			assertEquals(((EmailEpr) theEpr).getPassword(), "password");
		}
		catch (Exception ex)
		{			
			fail(ex.toString());
		}
	}
	
	public void testFtpEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);

		try
		{
			FTPEpr epr = new FTPEpr("http://www.foo.bar");
			
			epr.setPassive(true);
			
			msg.getHeader().getCall().setTo(epr);
			
			final String xmlRepresentation = msgToXML((MessageImpl)msg) ;
			log.debug("Document is "+xmlRepresentation);
			
			final MessageImpl nImpl = msgFromXML(xmlRepresentation) ;
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();
			
			assertEquals(theEpr instanceof FTPEpr, true);
			
			assertEquals(((FTPEpr) theEpr).getPassive(), true);
		}
		catch (Exception ex)
		{			
			fail(ex.toString());
		}
	}
	
	public void testJdbcEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);

		try
		{
			JDBCEpr epr = new JDBCEpr("http://www.foo.bar", "SOME FAKE SQL");
			
			msg.getHeader().getCall().setTo(epr);
			
			final String xmlRepresentation = msgToXML((MessageImpl)msg) ;
			log.debug("Document is "+xmlRepresentation);
			
			final MessageImpl nImpl = msgFromXML(xmlRepresentation) ;
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();
			
			assertEquals(theEpr instanceof JDBCEpr, true);
			
			assertEquals(((JDBCEpr) theEpr).getSQL(), "SOME FAKE SQL");
		}
		catch (Exception ex)
		{			
			fail(ex.toString());
		}
	}
	
	public void testFileEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);

		try
		{
			FileEpr epr = new FileEpr("file://tmp/bar.txt");
			
			epr.setErrorDelete(true);
			
			msg.getHeader().getCall().setTo(epr);
			
			final String xmlRepresentation = msgToXML((MessageImpl)msg) ;
			log.debug("Document is "+xmlRepresentation);
			
			final MessageImpl nImpl = msgFromXML(xmlRepresentation) ;
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();
			
			assertEquals(theEpr instanceof FileEpr, true);
			
			assertEquals(((FileEpr) theEpr).getErrorDelete(), true);
		}
		catch (Exception ex)
		{			
			fail(ex.toString());
		}
	}
	
	public void testSFtpEPRType ()
	{
		Message msg = MessageFactory.getInstance().getMessage(MessageType.JBOSS_XML);

		try
		{
			SFTPEpr epr = new SFTPEpr(new URL("http://www.foo.bar"), new URL("http://www.bar.foo"));

			assertEquals(epr.getCertificateURL().toString(), "http://www.bar.foo");
			
			msg.getHeader().getCall().setTo(epr);
			
			final String xmlRepresentation = msgToXML((MessageImpl)msg) ;
			log.debug("Document is "+xmlRepresentation);
			
			final MessageImpl nImpl = msgFromXML(xmlRepresentation) ;
			
			EPR theEpr = nImpl.getHeader().getCall().getTo();
			
			assertEquals(theEpr instanceof SFTPEpr, true);
			
			assertEquals(((SFTPEpr) theEpr).getCertificateURL().toString(), "http://www.bar.foo");
		}
		catch (Exception ex)
		{			
			fail(ex.toString());
		}
	}
	
	public static String msgToXML(final MessageImpl msg)
		throws Exception
	{
		return (String) Util.serialize(msg);
	}
	
	public static MessageImpl msgFromXML(final String xmlRepresentation)
		throws Exception
	{
        return (MessageImpl) Util.deserialize(xmlRepresentation);
	}
}
