/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners.message;

import junit.framework.TestCase;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.message.format.MessageFactory;

public class ActionProcessingPipelineUnitTest extends TestCase
{
	public void testProperty() {
		
	}
	
    @Override
    protected void setUp() throws Exception
    {
        super.setUp();
        MockActionInfo.reset();
        System.setProperty("javax.xml.registry.ConnectionFactoryClass","org.apache.ws.scout.registry.ConnectionFactoryImpl");

        com.arjuna.common.util.propertyservice.PropertyManager pm = ModulePropertyManager.getPropertyManager(ModulePropertyManager.DBSTORE_MODULE);
        pm.setProperty("javax.xml.registry.ConnectionFactoryClass", "org.apache.ws.scout.registry.ConnectionFactoryImpl");
    }
    
    /*
     * Tests to run
     *  - create a pipeline with a failure, make sure failure is called in correct order and subsequent actions are not called.
     *    Check initialise and destroy called.
     *  
     *  Do the above for each type of action.
     */
    public void testActionPipelineProcessor()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4) ;
        
        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList(), 4, 2, 0) ;
        checkOrder(MockActionInfo.getExceptionList()) ;
        
        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList(), 4, 2, 0, 4, 2, 0) ;
        checkOrder(MockActionInfo.getExceptionList()) ;
        
        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList(), 4, 2, 0) ;
    }
    
    public void testActionPipelineProcessorFailure()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;
        
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;
        addAction(configTree, MockActionPipelineProcessorFailure.class.getName(),
            null, null, null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, null, null) ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4) ;
        
        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 2, 0) ;

        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 0, 2) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 2, 0, 2, 0) ;

        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList(), 4, 2, 0) ;
    }
    
    public void testOverriddenActionPipelineProcessor()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;
        
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            "process", null, null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, "anotherProcessSuccess", null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
                null, null, "anotherProcessException") ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            "process, anotherProcess", "anotherProcessSuccess", "anotherProcessException") ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4, 6) ;
        
        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 7) ;
        checkOrder(MockActionInfo.getSuccessList(), 7, 4, 3, 0) ;
        checkOrder(MockActionInfo.getExceptionList()) ;
        
        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 7, 0, 2, 4, 6, 7) ;
        checkOrder(MockActionInfo.getSuccessList(), 7, 4, 3, 0, 7, 4, 3, 0) ;
        checkOrder(MockActionInfo.getExceptionList()) ;
        
        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList(), 6, 4, 2, 0) ;
    }
    
    public void testOverriddenActionPipelineProcessorFailure()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;
        
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            "process", null, null) ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            null, "anotherProcessSuccess", null) ;
        addAction(configTree, MockActionPipelineProcessorFailure.class.getName(),
                null, null, "anotherProcessException") ;
        addAction(configTree, MockActionPipelineProcessor.class.getName(),
            "process, anotherProcess", "anotherProcessSuccess", "anotherProcessException") ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4, 6) ;
        
        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 5, 2, 0) ;
        
        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 5, 2, 0, 5, 2, 0) ;
        
        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList(), 6, 4, 2, 0) ;
    }
    
    public void testOverriddenActionLifecycleProcessor()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;
        
        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
            "process", null, null) ;
        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
            null, "processSuccess", null) ;
        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
                null, null, "processException") ;
        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
            "process, anotherProcess", "anotherProcessSuccess", "anotherProcessException") ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4, 6) ;
        
        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 7) ;
        checkOrder(MockActionInfo.getSuccessList(), 7, 2) ;
        checkOrder(MockActionInfo.getExceptionList()) ;
        
        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 7, 0, 2, 4, 6, 7) ;
        checkOrder(MockActionInfo.getSuccessList(), 7, 2, 7, 2) ;
        checkOrder(MockActionInfo.getExceptionList()) ;
        
        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList(), 6, 4, 2, 0) ;
    }
    
    public void testOverriddenActionLifecycleProcessorFailure()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;
        
        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
            "process", null, null) ;
        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
            null, "anotherProcessSuccess", null) ;
        addAction(configTree, MockActionLifecycleProcessorFailure.class.getName(),
                null, null, "anotherProcessException") ;
        addAction(configTree, MockActionLifecycleProcessor.class.getName(),
            "process, anotherProcess", "anotherProcessSuccess", "anotherProcessException") ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList(), 0, 2, 4, 6) ;
        
        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 5) ;
        
        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 5, 5) ;
        
        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList(), 6, 4, 2, 0) ;
    }
    
    public void testOverriddenActionProcessor()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;
        
        addAction(configTree, MockActionProcessor.class.getName(),
            "process", null, null) ;
        addAction(configTree, MockActionProcessor.class.getName(),
            null, "processSuccess", null) ;
        addAction(configTree, MockActionProcessor.class.getName(),
                null, null, "processException") ;
        addAction(configTree, MockActionProcessor.class.getName(),
            "process, anotherProcess", "anotherProcessSuccess", "anotherProcessException") ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList()) ;
        
        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 7) ;
        checkOrder(MockActionInfo.getSuccessList(), 7, 2) ;
        checkOrder(MockActionInfo.getExceptionList()) ;
        
        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertTrue("Pipeline process failure", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 7, 8, 10, 12, 14, 15) ;
        checkOrder(MockActionInfo.getSuccessList(), 7, 2, 15, 10) ;
        checkOrder(MockActionInfo.getExceptionList()) ;
        
        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList()) ;
    }
    
    public void testOverriddenActionProcessorFailure()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;
        
        addAction(configTree, MockActionProcessor.class.getName(),
            "process", null, null) ;
        addAction(configTree, MockActionProcessor.class.getName(),
            null, "anotherProcessSuccess", null) ;
        addAction(configTree, MockActionProcessorFailure.class.getName(),
                null, null, "anotherProcessException") ;
        addAction(configTree, MockActionProcessor.class.getName(),
            "process, anotherProcess", "anotherProcessSuccess", "anotherProcessException") ;

        final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(configTree) ;
        pipeline.initialise() ;
        checkOrder(MockActionInfo.getInitialiseList()) ;
        
        final boolean result = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", result) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 5) ;
        
        final boolean secondResult = pipeline.process(MessageFactory.getInstance().getMessage()) ;
        assertFalse("Pipeline process succeeded", secondResult) ;
        checkOrder(MockActionInfo.getProcessList(), 0, 2, 4, 6, 8, 10) ;
        checkOrder(MockActionInfo.getSuccessList()) ;
        checkOrder(MockActionInfo.getExceptionList(), 5, 11) ;
        
        pipeline.destroy() ;
        checkOrder(MockActionInfo.getDestroyList()) ;
    }
    
    public void testErrorActionProcessorException() throws Exception
	{
		final ConfigTree configTree = new ConfigTree("parent");
		configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
				ListenerTagNames.MEP_ONE_WAY) ;

		addAction(configTree, ErrorActionProcessor.class.getName(), "process",
				null, null);
		addAction(configTree, ErrorActionProcessor.class.getName(), null,
				"anotherProcessSuccess", null);
		addAction(configTree, MockActionProcessorFailure.class.getName(), null,
				null, "anotherProcessException");
		addAction(configTree, ErrorActionProcessor.class.getName(),
				"process, anotherProcess", "anotherProcessSuccess",
				"anotherProcessException");

		final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(
				configTree);
		pipeline.initialise();
		
		final boolean result = pipeline.process(MessageFactory.getInstance()
				.getMessage());
		
		assertFalse(result);
	}
    
    public void testNPEActionProcessorException() throws Exception
	{
		final ConfigTree configTree = new ConfigTree("parent");
		configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
				ListenerTagNames.MEP_ONE_WAY) ;

		addAction(configTree, NPEActionProcessor.class.getName(), "process",
				null, null);
		addAction(configTree, NPEActionProcessor.class.getName(), null,
				"anotherProcessSuccess", null);
		addAction(configTree, MockActionProcessorFailure.class.getName(), null,
				null, "anotherProcessException");
		addAction(configTree, ErrorActionProcessor.class.getName(),
				"process, anotherProcess", "anotherProcessSuccess",
				"anotherProcessException");

		final ActionProcessingPipeline pipeline = new ActionProcessingPipeline(
				configTree);
		pipeline.initialise();
		
		final boolean result = pipeline.process(MessageFactory.getInstance()
				.getMessage());
		
		assertFalse(result);
	}
    
    public void testInvalidConstructor()
        throws Exception
    {
        final ConfigTree configTree = new ConfigTree("parent") ;
        configTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG,
        		ListenerTagNames.MEP_ONE_WAY) ;
        
        addAction(configTree, MockActionPipelineProcessorConstructorFailure.class.getName(),
            null, null, null) ;
        try
        {
            new ActionProcessingPipeline(configTree) ;
            fail("Expected ConfigurationException") ;
        }
        catch (final ConfigurationException ce)
        {
            
        }
    }

    public static void addAction(final ConfigTree configTree, final String actionName) {
        addAction(configTree, actionName, null, null, null);
    }
    public static void addAction(final ConfigTree configTree, final String actionName,
        final String processOverride, final String successOverride,
        final String exceptionOverride)
    {
        final ConfigTree actionChild = new ConfigTree(ListenerTagNames.ACTION_ELEMENT_TAG, configTree) ;
        actionChild.setAttribute(ListenerTagNames.ACTION_CLASS_TAG, actionName) ;
        if (processOverride != null)
        {
            actionChild.setAttribute(ListenerTagNames.PROCESS_METHOD_TAG, processOverride) ;
        }
        if (successOverride != null)
        {
            actionChild.setAttribute(ListenerTagNames.NORMAL_COMPLETION_METHOD_TAG, successOverride) ;
        }
        if (exceptionOverride != null)
        {
            actionChild.setAttribute(ListenerTagNames.EXCEPTION_METHOD_TAG, exceptionOverride) ;
        }
    }
    
    private void checkOrder(final Integer[] list, int ... values)
    {
        final int numValues = (values == null ? 0 : values.length) ;
        final int listLength = (list == null ? 0 : list.length) ;
        
        assertEquals("Unexpected list/values count", numValues, listLength) ;
        
        for(int count = 0 ; count < numValues ; count++)
        {
            assertEquals("Unexpected call order at count " + count, values[count], list[count].intValue()) ;
        }
    }
}
