/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.gateway;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Enumeration;
import java.util.HashMap;

import javax.jms.BytesMessage;
import javax.jms.Destination;
import javax.jms.JMSException;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.junit.Before;
import org.junit.Test;
import org.mockejb.jms.MockQueue;
import org.mockejb.jms.ObjectMessageImpl;
import org.mockejb.jms.TextMessageImpl;

/**
 * Unit test for PackageJmsMessageContents
 * 
 * @author <a href="daniel.bevenius@redpill.se">Daniel Bevenius</a>				
 *
 */
public class PackageJmsMessageContentsUnitTest
{
	@SuppressWarnings ( "unused" )
	private Logger log = Logger.getLogger( PackageJmsMessageContentsUnitTest.class );
	
	private final static String messageContent = "Test Message Content";
	private final static String jmsMessageID = "123456780";
	private final static String jmsCorrelationID = "YYXX-123456780-GG";
	
	private PackageJmsMessageContents packer;
	private ObjectMessageImpl objectMsg;
	
	@Before
	public void setup()
	{
        ConfigTree emptyConfig = new ConfigTree("empty");
        MessagePayloadProxy proxy = PackageJmsMessageContents.createPayloadProxy(emptyConfig);
        
        packer = new PackageJmsMessageContents(proxy);
		objectMsg = new ObjectMessageImpl();
	}
	
	@Test
	public void process_ObjectMessage() throws JMSException, IOException, URISyntaxException, MessageDeliverException {
		objectMsg.setObject( messageContent );
		Message message = packer.process( objectMsg );
		
		assertThatByteArrayHasBeenSet( message );
		
		Object object = message.getBody().get();
		assertTrue ( object instanceof String );
		assertEquals (  String.class.getName(), object.getClass().getName() );
		
		assertEquals ( messageContent, message.getBody().get());
	}
	
	@Test
	public void process_ObjectMessage_HashMap() throws JMSException, IOException, URISyntaxException, MessageDeliverException {
		final String key = "testkey";
		final String value = "testvalue";
		HashMap<String,String> hashMap = new HashMap<String,String>();
		hashMap.put( key, value );
		objectMsg.setObject( hashMap );
		Message message = packer.process( objectMsg );
		
		assertThatByteArrayHasBeenSet( message );
		
		Object object = message.getBody().get();
		assertTrue ( object instanceof HashMap );
		
		assertEquals (  HashMap.class.getName(), object.getClass().getName() );
		HashMap actualMap = (HashMap) object;
		assertEquals ( hashMap, actualMap );
	}
	
	@Test
	public void process_TextMessage() throws JMSException, IOException, URISyntaxException, MessageDeliverException {
		TextMessageImpl textMessage = new TextMessageImpl();
		textMessage.setText( messageContent );
		Message message = packer.process( textMessage );
		
		assertThatByteArrayHasBeenSet( message );
		
		Object object = message.getBody().get();
		assertTrue ( object instanceof String );
		final String actualContent = (String) object;
		assertEquals ( messageContent, actualContent );
	}
	
	@Test
	public void process_BytesMessage() throws JMSException, IOException, URISyntaxException, MessageDeliverException {
		MockBytesMessage byteMessage = new MockBytesMessage();
		
		Message message = packer.process( byteMessage );
		
		assertThatByteArrayHasBeenSet( message );
		
	}
	
	@Test
	public void process_with_JMSMessageID() throws JMSException, IOException, URISyntaxException, MessageDeliverException {
		objectMsg.setObject( messageContent );
		objectMsg.setJMSMessageID( jmsMessageID );
		
		Message message = packer.process( objectMsg );
		
		final String actualContent = (String)message.getBody().get();
		assertEquals ( messageContent, actualContent );
		assertEquals ( new URI( jmsMessageID ), message.getHeader().getCall().getMessageID()  );
	}
	
	@Test
	public void process_with_JMSCorrelationID() throws JMSException, IOException, URISyntaxException, MessageDeliverException {
		objectMsg.setObject( messageContent );
		objectMsg.setJMSCorrelationID( jmsCorrelationID );
		
		Message message = packer.process( objectMsg );
		
		assertEquals ( jmsCorrelationID, message.getHeader().getCall().getRelatesTo().getFragment()  );
	}
	
	@Test
	public void process_with_Properties() throws JMSException, IOException, URISyntaxException, MessageDeliverException {
		final String propertyKey = "myProperty";
		final String propertyValue = "myPropertyValue";
		objectMsg.setObject( messageContent );
		objectMsg.setObjectProperty( propertyKey, propertyValue );
		
		Message message = packer.process( objectMsg );
		
		assertEquals ( propertyValue, message.getProperties().getProperty( propertyKey ) );
	}
	
	@Test
	public void process_with_JMSReplyTo() throws JMSException, IOException, URISyntaxException, MessageDeliverException {
		objectMsg.setObject( messageContent );
		MockQueue jmsReplyToQueue = new MockQueue( "mockReplyToQueueName");
		objectMsg.setJMSReplyTo( jmsReplyToQueue );
		
		Message message = packer.process( objectMsg );
		
		EPR replyTo = message.getHeader().getCall().getReplyTo();
		assertTrue(  replyTo instanceof JMSEpr );
		JMSEpr jmsEpr = (JMSEpr) replyTo;
		assertEquals ( jmsReplyToQueue.getQueueName() , jmsEpr.getDestinationName() );
	}
	
	/**
	 * This asserts that the JMS Message's bodies content
	 * is always set on the ESB Message object as a byte array.
	 * 
	 * This is a backward compability issue and more on the reason for
	 * this can be found in this ESB User Forum thread:
	 * http://www.jboss.com/index.html?module=bb&op=viewtopic&t=113726&postdays=0&postorder=asc&start=10
	 *
	 */
	private void assertThatByteArrayHasBeenSet( Message esbMessage )
	{
		assertNotNull ("byte array must always be set regardless of the type of JMS Message", esbMessage.getBody().get() );
	}
	
	/**
	 * Just here to get Ant to find annotated test.
	 */
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( PackageJmsMessageContentsUnitTest.class );
	}

	private static class MockBytesMessage implements BytesMessage
	{

		public long getBodyLength() throws JMSException
		{
			return 0;
		}

		public boolean readBoolean() throws JMSException
		{
			return false;
		}

		public byte readByte() throws JMSException
		{
			return 0;
		}

		public int readBytes( byte[] arg0 ) throws JMSException
		{
			return -1;
		}

		public int readBytes( byte[] arg0, int arg1 ) throws JMSException
		{
			return 0;
		}

		public char readChar() throws JMSException
		{
			return 0;
		}

		public double readDouble() throws JMSException
		{
			return 0;
		}

		public float readFloat() throws JMSException
		{
			return 0;
		}

		public int readInt() throws JMSException
		{
			return 0;
		}

		public long readLong() throws JMSException
		{
			return 0;
		}

		public short readShort() throws JMSException
		{
			return 0;
		}

		public String readUTF() throws JMSException
		{
			return null;
		}

		public int readUnsignedByte() throws JMSException
		{
			return 0;
		}

		public int readUnsignedShort() throws JMSException
		{
			return 0;
		}

		public void reset() throws JMSException
		{
		}

		public void writeBoolean( boolean arg0 ) throws JMSException
		{
		}

		public void writeByte( byte arg0 ) throws JMSException
		{
		}

		public void writeBytes( byte[] arg0 ) throws JMSException
		{
		}

		public void writeBytes( byte[] arg0, int arg1, int arg2 )
				throws JMSException
		{
		}

		public void writeChar( char arg0 ) throws JMSException
		{
		}

		public void writeDouble( double arg0 ) throws JMSException
		{
		}

		public void writeFloat( float arg0 ) throws JMSException
		{
		}

		public void writeInt( int arg0 ) throws JMSException
		{
		}

		public void writeLong( long arg0 ) throws JMSException
		{
		}

		public void writeObject( Object arg0 ) throws JMSException
		{
		}

		public void writeShort( short arg0 ) throws JMSException
		{
		}

		public void writeUTF( String arg0 ) throws JMSException
		{
		}

		public void acknowledge() throws JMSException
		{
		}

		public void clearBody() throws JMSException
		{
		}

		public void clearProperties() throws JMSException
		{
		}

		public boolean getBooleanProperty( String arg0 ) throws JMSException
		{
			return false;
		}

		public byte getByteProperty( String arg0 ) throws JMSException
		{
			return 0;
		}

		public double getDoubleProperty( String arg0 ) throws JMSException
		{
			return 0;
		}

		public float getFloatProperty( String arg0 ) throws JMSException
		{
			return 0;
		}

		public int getIntProperty( String arg0 ) throws JMSException
		{
			return 0;
		}

		public String getJMSCorrelationID() throws JMSException
		{
			return null;
		}

		public byte[] getJMSCorrelationIDAsBytes() throws JMSException
		{
			return null;
		}

		public int getJMSDeliveryMode() throws JMSException
		{
			return 0;
		}

		public Destination getJMSDestination() throws JMSException
		{
			return null;
		}

		public long getJMSExpiration() throws JMSException
		{
			return 0;
		}

		public String getJMSMessageID() throws JMSException
		{
			return null;
		}

		public int getJMSPriority() throws JMSException
		{
			return 0;
		}

		public boolean getJMSRedelivered() throws JMSException
		{
			return false;
		}

		public Destination getJMSReplyTo() throws JMSException
		{
			return null;
		}

		public long getJMSTimestamp() throws JMSException
		{
			return 0;
		}

		public String getJMSType() throws JMSException
		{
			return null;
		}

		public long getLongProperty( String arg0 ) throws JMSException
		{
			return 0;
		}

		public Object getObjectProperty( String arg0 ) throws JMSException
		{
			return null;
		}

		public Enumeration getPropertyNames() throws JMSException
		{
			return null;
		}

		public short getShortProperty( String arg0 ) throws JMSException
		{
			return 0;
		}

		public String getStringProperty( String arg0 ) throws JMSException
		{
			return null;
		}

		public boolean propertyExists( String arg0 ) throws JMSException
		{
			return false;
		}

		public void setBooleanProperty( String arg0, boolean arg1 )
				throws JMSException
		{
		}

		public void setByteProperty( String arg0, byte arg1 )
				throws JMSException
		{
		}

		public void setDoubleProperty( String arg0, double arg1 )
				throws JMSException
		{
		}

		public void setFloatProperty( String arg0, float arg1 )
				throws JMSException
		{
		}

		public void setIntProperty( String arg0, int arg1 ) throws JMSException
		{
		}

		public void setJMSCorrelationID( String arg0 ) throws JMSException
		{
		}

		public void setJMSCorrelationIDAsBytes( byte[] arg0 )
				throws JMSException
		{
		}

		public void setJMSDeliveryMode( int arg0 ) throws JMSException
		{
		}

		public void setJMSDestination( Destination arg0 ) throws JMSException
		{
		}

		public void setJMSExpiration( long arg0 ) throws JMSException
		{
		}

		public void setJMSMessageID( String arg0 ) throws JMSException
		{
		}

		public void setJMSPriority( int arg0 ) throws JMSException
		{
		}

		public void setJMSRedelivered( boolean arg0 ) throws JMSException
		{
		}

		public void setJMSReplyTo( Destination arg0 ) throws JMSException
		{
		}

		public void setJMSTimestamp( long arg0 ) throws JMSException
		{
		}

		public void setJMSType( String arg0 ) throws JMSException
		{
		}

		public void setLongProperty( String arg0, long arg1 )
				throws JMSException
		{
		}

		public void setObjectProperty( String arg0, Object arg1 )
				throws JMSException
		{
		}

		public void setShortProperty( String arg0, short arg1 )
				throws JMSException
		{
		}

		public void setStringProperty( String arg0, String arg1 )
				throws JMSException
		{
		}
	}
}
