/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.converters;

import java.util.ArrayList;
import java.util.List;

import junit.framework.TestCase;

import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.actions.TestBean;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.io.xml.DomDriver;

/**
 * ObjectToXStream unit tests.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 * @since Version 4.0
 */
public class ObjectToXStreamUnitTest extends TestCase {

    public void test_default() throws ActionProcessingException {
        ConfigTree config = new ConfigTree("config");

        ObjectToXStream objectToXStream = new ObjectToXStream(config);
        
        Message oMsg = MessageFactory.getInstance().getMessage();
        oMsg.getBody().add(new TestBean("Tom", "1234"));
        oMsg = objectToXStream.process(oMsg);
        String xml = (String)oMsg.getBody().get();
        XStream xstream = new XStream(new DomDriver());
        TestBean bean = new TestBean();

        assertTrue("Expected [<TestBean>].  Got [" + xml + "]", xml.startsWith("<TestBean>"));
        xstream.alias("TestBean", TestBean.class);
        xstream.fromXML(xml, bean);
        
        assertEquals("Tom", bean.getName());
        assertEquals("1234", bean.getPhone());
    }

    public void test_with_package() throws ActionProcessingException {
        ConfigTree config = new ConfigTree("config");
        
        config.setAttribute("exclude-package", "false");
        ObjectToXStream objectToXStream = new ObjectToXStream(config);
        
        Message oMsg = MessageFactory.getInstance().getMessage();
        oMsg.getBody().add(new TestBean("Tom", "1234"));
        objectToXStream.process(oMsg);
        String xml = (String)oMsg.getBody().get();

        XStream xstream = new XStream(new DomDriver());
        TestBean bean = new TestBean();

        assertTrue("Expected start with [<" + TestBean.class.getName() + ">].  Got [" + xml + "]", xml.startsWith("<" + TestBean.class.getName() + ">"));
        xstream.fromXML(xml, bean);
        
        assertEquals("Tom", bean.getName());
        assertEquals("1234", bean.getPhone());
    }

    public void test_with_alias() throws ActionProcessingException {
        ConfigTree config = new ConfigTree("config");

        config.setAttribute("class-alias", "TomsClass");
        ObjectToXStream objectToXStream = new ObjectToXStream(config);
        
        Message oMsg = MessageFactory.getInstance().getMessage();
        oMsg.getBody().add(new TestBean("Tom", "1234"));
        objectToXStream.process(oMsg);
        String xml = (String)oMsg.getBody().get();

        XStream xstream = new XStream(new DomDriver());
        xstream.alias("TomsClass", TestBean.class);
        TestBean bean = new TestBean();

        assertTrue("Expected start with [<TomsClass>].  Got [" + xml + "]", xml.startsWith("<TomsClass>"));
        xstream.fromXML(xml, bean);
        
        assertEquals("Tom", bean.getName());
        assertEquals("1234", bean.getPhone());
    }
}
