/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.util;

import java.io.File;
import java.io.IOException;
import java.util.List;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;

import com.enterprisedt.net.ftp.FTPClient;
import com.enterprisedt.net.ftp.FTPConnectMode;
import com.enterprisedt.net.ftp.FTPException;
import com.enterprisedt.net.ftp.FTPTransferType;

/**
 * Simplified FTP transfers
 * <p>
 * Description: Implements a simple set of FTP functionality Parameters to
 * establish the FTP connection are provided at construction time and cannot
 * change during the lifetime of the object <br/>Hides low level details.
 * Current implementation is based on the "Entreprise Distributed Technology
 * edtFTPj" library but this can be changed with no impact to existing code,
 * just by changing this class without modifying the signature of it's public
 * methods
 * </p>
 */

public class FtpClientUtil
{
	public static final String PARMS_FTP_SERVER = "ftpServer";

	public static final String PARMS_USER = "ftpUser";

	public static final String PARMS_PASSWD = "ftpPassword";

	public static final String PARMS_PORT = "ftpPort";

	public static final String PARMS_REMOTE_DIR = "ftpRemoteDir";

	public static final String PARMS_LOCAL_DIR = "ftpLocalDir";

	public static final String PARMS_ASCII = "ftpAscii";

	public static final String PARMS_PASSIVE = "ftpPassive";

	private static final String TMP_SUFFIX = ".rosettaPart";

	public enum XFER_TYPE
	{
		ascii, binary
	};

	private ConfigTree m_oParms;

	private String m_sFtpServer, m_sUser, m_sPasswd;

	private String m_sRemoteDir, m_sLocalDir;

	private int m_iPort;

	private boolean m_bPassive;

	public String getRemoteDir ()
	{
		return m_sRemoteDir;
	}

	private FTPClient m_oConn = new FTPClient();

	private FTPTransferType m_oXferType = FTPTransferType.BINARY;

	/**
	 * Checks validity and completeness of parameters, and keeps the info
	 * internally for subsequent FTP requests
	 * 
	 * @param p_oP
	 *            ConfigTree
	 * @throws ConfigurationException :
	 *             if parameters are invalid or incomplete
	 *             <li>Parameters: (XML attributes at the root level) </li>
	 *             <li> ftpServer = name or IP of FTP server </li>
	 *             <li> ftpUser = login ID for server </li>
	 *             <li> ftpPassword </li>
	 *             <li> localDirURI = absolute path in the local filesystem
	 *             </li>
	 *             <li> remoteDirURI = remote path is relative to ftp user home
	 *             in remote computer </li>
	 * @throws RemoteFileSystemException
	 */

	public FtpClientUtil (ConfigTree p_oP, boolean p_bConnect) throws RemoteFileSystemException, ConfigurationException
	{
		m_oParms = p_oP;
		initialize(p_bConnect);
	} // _________________________________

	public FtpClientUtil (List<KeyValuePair> attribs, boolean connect)
			throws RemoteFileSystemException, ConfigurationException
	{
		m_oParms = new ConfigTree("fromProps");
		for (KeyValuePair oCurr : attribs)
			m_oParms.setAttribute(oCurr.getKey(), oCurr.getValue());
		initialize(connect);
	} // __________________________________

	private void initialize (boolean bConnect) throws RemoteFileSystemException, ConfigurationException
	{
		checkParms();
		
		try
		{
			if (bConnect)
			{
				m_oConn.setRemoteHost(m_sFtpServer);
				m_oConn.setRemotePort(m_iPort);
				m_oConn.connect();
				
				for (int i1 = 0; i1 < 10 && !m_oConn.connected(); i1++)
				{
					try
					{
						// TODO magic number
						Thread.sleep(200);
					}
					catch (InterruptedException ex)
					{
					}
				}
				
				if (!m_oConn.connected())
					throw new RemoteFileSystemException("Can't connect to FTP server");
				m_oConn.user(m_sUser);
				m_oConn.password(m_sPasswd);
				m_oConn
						.setConnectMode((m_bPassive) ? FTPConnectMode.PASV : FTPConnectMode.ACTIVE);
			}
		}
		catch (RemoteFileSystemException ex)
		{
			throw ex;
		}
		catch (Exception ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	} // __________________________________

	/**
	 * Terminates ftp session and frees resources
	 * <li>Well behaved programs should make sure to call this method </li>
	 */
	public void quit ()
	{
		if (null != m_oConn) try
		{
			m_oConn.quit();
		}
		catch (Exception e)
		{
		}
	} // _________________________________

	/**
	 * Deletes specified file in remote directory
	 * 
	 * @param p_sFile
	 *            String : filename to delete. Method will attempt to delete
	 *            file with rightmost node of argument within remote directory
	 *            specified in 'remoteDirURI'
	 * @throws RemoteFileSystemException :
	 *             if ftp connection cannot be established, or file cannot be
	 *             deleted in remote directory
	 */
	public void deleteRemoteFile (String p_sFile) throws RemoteFileSystemException
	{
		try
		{
			m_oConn.delete(getRemoteDir() + "/" + new File(p_sFile).getName());
		}
		catch (FTPException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
		catch (IOException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	} // _________________________________

	public void remoteDelete (File p_oFile) throws RemoteFileSystemException
	{
		try
		{
			m_oConn.delete(fileToFtpString(p_oFile));
		}
		catch (FTPException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
		catch (IOException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	} // _________________________________

	/**
	 * Gets the list of files in the remote directory that end with arg0
	 * 
	 * @param p_sSuffix
	 *            String : retrieve only files that end with that suffix - all
	 *            files if null
	 * @throws RemoteFileSystemException :
	 *             if ftp connection cannot be established, or problems
	 *             encountered
	 */
	public String[] getFileListFromRemoteDir (String p_sSuffix)
			throws RemoteFileSystemException
	{
		String sSuffix = (null == p_sSuffix) ? "*" : "*" + p_sSuffix;
		
		try
		{
			return m_oConn.dir(sSuffix);
		}
		catch (FTPException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
		catch (IOException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	} // _________________________________

	/**
	 * Change remote directory
	 * 
	 * @param p_sDir
	 *            String : directory to set
	 * @throws RemoteFileSystemException :
	 *             if ftp connection cannot be established, or problems
	 *             encountered
	 */
	public void setRemoteDir (String p_sDir) throws RemoteFileSystemException
	{
		try
		{
			m_oConn.chdir(p_sDir);
		}
		catch (FTPException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
		catch (IOException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	} // _________________________________

	/**
	 * Renames specified file in remote directory to specified new name
	 * 
	 * @param p_sFrom
	 *            String : filename to rename
	 * @param p_sTo
	 *            String : new filename
	 * @throws RemoteFileSystemException :
	 *             if ftp connection cannot be established, or file cannot be
	 *             renamed to new name in remote directory
	 *             <li>Method will attempt to rename file with rightmost node
	 *             of argument within remote directory specified in
	 *             'remoteDirURI', to new name inside the SAME remote directory
	 */
	public void renameInRemoteDir (String p_sFrom, String p_sTo)
			throws RemoteFileSystemException
	{
		String sRmtFrom = new File(p_sFrom).getName();
		String sRmtTo = new File(p_sTo).getName();

		try
		{
			m_oConn.rename(getRemoteDir() + "/" + sRmtFrom,
					getRemoteDir() + "/" + sRmtTo);
		}
		catch (Exception e)
		{
			String sMess = this.getClass().getSimpleName() + " can't rename in remote directory <" + e
					.getMessage() + ">";
			throw new RemoteFileSystemException(sMess);
		}
	} // _________________________________

	public void remoteRename (File p_oFrom, File p_oTo) throws RemoteFileSystemException
	{
		try
		{
			m_oConn.rename(fileToFtpString(p_oFrom), fileToFtpString(p_oTo));
		}
		catch (Exception e)
		{
			String sMess = this.getClass().getSimpleName() + " can't rename in remote directory <" + e
					.getMessage() + ">";
			throw new RemoteFileSystemException(sMess);
		}
	} // _________________________________

	/**
	 * Uploads specified file from local directory (localDirURI) to remote
	 * directory (remoteDirURI)
	 * 
	 * @param p_oFile
	 *            String : filename to upload
	 * @throws RemoteFileSystemException :
	 *             if ftp connection cannot be established, or file cannot be
	 *             uploaded
	 *             <li> local file will be renamed during transfer
	 *             ('.xferNotReady' appended to name)</li>
	 *             <li> upon successful completion. the suffix '.xferDone' will
	 *             be appended to the original filename </li>
	 */
	public void uploadFile (File p_oFile, String p_sRemoteName)
			throws RemoteFileSystemException
	{
		String sRemoteOK = getRemoteDir() + "/" + p_sRemoteName;
		String sRemoteTmp = sRemoteOK + TMP_SUFFIX;
		
		try
		{
			m_oConn.setType(m_oXferType);
			m_oConn.put(fileToFtpString(p_oFile), sRemoteTmp);
			m_oConn.rename(sRemoteTmp, sRemoteOK);
		}
		catch (FTPException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
		catch (IOException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	} // _________________________________

	/**
	 * Downloads specified file from remote directory (remoteDirURI) to local
	 * directory (localDirURI)
	 * 
	 * @param p_sFile
	 *            String : filename to download
	 * @throws RemoteFileSystemException :
	 *             if ftp connection cannot be established, or file cannot be
	 *             downloaded
	 *             <li> local file is assigned a temporary name during transfer
	 *             </li>
	 *             <li> upon successful completion, local temporary file will be
	 *             renamed to name specified in argument, and suffix '.xferDone'
	 *             will be appended to the original filename in the remote
	 *             directory </li>
	 */
	public void downloadFile (String p_sFile, String p_sFinalName)
			throws RemoteFileSystemException
	{
		try
		{
			File oLocalDir = new File(m_sLocalDir);
			File oLclFile = File.createTempFile("Rosetta_", TMP_SUFFIX, oLocalDir);
	
			try
			{
				oLclFile.delete();
			}
			catch (Exception e)
			{
			}
			
			m_oConn.setType(m_oXferType);
			m_oConn.get(fileToFtpString(oLclFile), p_sFile);
	
			File oNew = new File(oLocalDir, p_sFinalName);
			
			if (oNew.exists()) 
				oNew.delete();
			
			FileUtil.renameTo(oLclFile, oNew);
		}
		catch (FTPException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
		catch (IOException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	} // _________________________________

	// Beware !!! The logic here seems wrong, but it works
	// It appears that there's some kind of bug in the edtftpj.jar library
	// The !XFER_TYPE.ascii.equals(p_oMode) should NOT be negated
	// But it works when negated (newlines from Unix to DOS)
	private void setXferType (XFER_TYPE p_oMode)
	{
		m_oXferType = !XFER_TYPE.ascii.equals(p_oMode) ? FTPTransferType.ASCII : FTPTransferType.BINARY;
	} // _________________________________

	private void checkParms () throws ConfigurationException
	{
		m_sFtpServer = m_oParms.getAttribute(PARMS_FTP_SERVER);
		if (null == m_sFtpServer)
			throw new ConfigurationException("No FTP server specified");

		m_sUser = m_oParms.getAttribute(PARMS_USER);
		if (null == m_sUser)
			throw new ConfigurationException("No username specified for FTP");

		m_sPasswd = m_oParms.getAttribute(PARMS_PASSWD);
		if (null == m_sPasswd)
			throw new ConfigurationException("No password specified for FTP");

		m_sRemoteDir = m_oParms.getAttribute(PARMS_REMOTE_DIR);
		if (null == m_sRemoteDir) 
			m_sRemoteDir = "";

		m_sLocalDir = m_oParms.getAttribute(PARMS_LOCAL_DIR);
		if (null == m_sLocalDir) 
			m_sLocalDir = ".";

		String sAux = m_oParms.getAttribute(PARMS_PORT);
		m_iPort = (null == sAux) ? 21 : Integer.parseInt(sAux);

		boolean bAscii = false;
		sAux = m_oParms.getAttribute(PARMS_ASCII);
		
		if (null != sAux) 
			bAscii = Boolean.parseBoolean(sAux);
		
		setXferType((bAscii) ? XFER_TYPE.ascii : XFER_TYPE.binary);

		m_bPassive = false;
		sAux = m_oParms.getAttribute(PARMS_PASSIVE);
		m_bPassive = (null != sAux) && Boolean.parseBoolean(sAux);
	} // __________________________________

	public static String fileToFtpString (File p_oF)
	{
		return (null == p_oF) ? null : p_oF.toString().replace("\\", "/");
	} // ________________________________

} // ____________________________________________________________________________
