package org.jboss.soa.esb.services.persistence;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

import java.net.URI;
import java.util.Map;

import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.urigen.MessageURIGenerator;

public interface MessageStore
{
    public final static String DEFAULT = "org.jboss.internal.soa.esb.persistence.format.db.DBMessageStoreImpl";
    public final static String CLASSIFICATION          = "org.jboss.soa.esb.messagestore.classification";
    public final static String CLASSIFICATION_DEFAULT  = "STORE";
    public final static String CLASSIFICATION_DLQ      = "DLQ";
    public final static String CLASSIFICATION_RDLVR    = "RDLVR";
    public final static String MESSAGE_URI             = "org.jboss.soa.esb.messagestore.message-uri";
    
 
    /**
	 * Get the implementation-specific message URI generator.
	 * @return the generator
	 */
	public MessageURIGenerator getMessageURIGenerator();
	
	/**
	 * Add a message to the store.
	 * @param message the message to store
	 * @return the URI of the generated message
	 * @throws MessageStoreException
	 */
	public URI addMessage (Message message, String classification) throws MessageStoreException;

	/**
	 * Get a message from the store by URI.
	 * @param uid the unique message id
	 * @return the message or null if it does not exist
	 * @throws MessageStoreException
	 */
	public Message getMessage (URI uid) throws MessageStoreException;
	
	/**
	 * Indicate that a message has not been delivered.
	 * @param uid
	 * @throws MessageStoreException
	 */
	public void setUndelivered(URI uid) throws MessageStoreException;
	
	/**
	 * Indicate that a message has been delivered.
	 * @param uid the unique message id
	 * @throws MessageStoreException
	 */
	public void setDelivered(URI uid) throws MessageStoreException;
	
	/**
	 * Get a map of undelivered messages.
	 * @return Map<URI, Message>
	 * @throws MessageStoreException
	 */
	public Map<URI, Message> getUndeliveredMessages(String classification) throws MessageStoreException;

    /**
     * Get a map of all messages for a certain classification.
     * @return map
     * @param classification
     * @throws MessageStoreException
     */
    public Map<URI, Message> getAllMessages(String classification) throws MessageStoreException;
   
    /**
     * Get a message from the store by URI.
     * @param uid the unique message id
     * @param classification the classification of the message
     * @return the message or null if it does not exist
     * @throws MessageStoreException
     */
    public Message getMessage (URI uid, String classification) throws MessageStoreException;
    /**
     * Removes a message from the store by URI.
     * @param uid the unique message id
     * @param classification the classification of the message
     * @return int response from the SQL delete command
     * @throws MessageStoreException
     */
    public int removeMessage (URI uid, String classification) throws MessageStoreException;
}