/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners.message;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionLifecycle;
import org.jboss.soa.esb.actions.ActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingFaultException;
import org.jboss.soa.esb.actions.BeanConfiguredAction;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.addressing.eprs.LogicalEPR;
import org.jboss.soa.esb.addressing.util.DefaultFaultTo;
import org.jboss.soa.esb.addressing.util.DefaultReplyTo;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.couriers.Courier;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierFactory;
import org.jboss.soa.esb.couriers.CourierUtil;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.message.errors.Factory;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.Properties;
import org.jboss.soa.esb.services.persistence.MessageStore;
import org.jboss.soa.esb.util.ClassUtil;

import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Action Processing Pipeline. <p/> Runs a list of action classes on a message
 * 
 * @author <a
 *         href="mailto:schifest@heuristica.com.ar">schifest@heuristica.com.ar</a>
 * @author kevin
 * @since Version 4.0
 */
public class ActionProcessingPipeline
{
	/**
	 * The logger instance.
	 */
	private final static Logger LOGGER = Logger
			.getLogger(ActionProcessingPipeline.class);

	/**
	 * The processors.
	 */
	private final ActionPipelineProcessor[] processors;

	/**
	 * The active flag.
	 */
	private final AtomicBoolean active = new AtomicBoolean(false);

	/**
	 * 
	 */
	private ServiceMessageCounter serviceMessageCounter;
	
	/**
	 * The transactional flag.
	 */
	private boolean transactional ;
	
	/**
	 * The flag indicating an action pipeline for a one way MEP.
	 */
	private boolean oneWay ;
	
        /**
         * The flag indicating whether we are using implicit or explicit processing.
         */
	private boolean defaultProcessing ;
	
	/**
	 * public constructor
	 * 
	 * @param config
	 *            The pipeline configuration.
	 */
	public ActionProcessingPipeline(final ConfigTree config)
			throws ConfigurationException
	{
		if (config == null)
		{
			throw new IllegalArgumentException(
					"Configuration needed for action classes");
		}

		final String mep = config.getAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG) ;
		final boolean oneWay ;
		final boolean defaultProcessing ;
		if (mep == null)
		{
		    oneWay = false ;
		    defaultProcessing = true ;
		}
		else if (ListenerTagNames.MEP_ONE_WAY.equals(mep))
		{
		    oneWay = true ;
		    defaultProcessing = false ;
		}
		else if (ListenerTagNames.MEP_REQUEST_RESPONSE.equals(mep))
		{
		    oneWay = false ;
		    defaultProcessing = false ;
		}
		else
		{
		    throw new ConfigurationException("Unrecognised action MEP: " + mep) ;
		}
		
		if (LOGGER.isDebugEnabled())
		{
		    LOGGER.debug("Using mep: " + mep + ", oneWay: " + oneWay + ", defaultProcessing: " + defaultProcessing) ;
		}
                this.oneWay = oneWay ;
                this.defaultProcessing = defaultProcessing ;
		
		final ConfigTree[] actionList = config
				.getChildren(ListenerTagNames.ACTION_ELEMENT_TAG);

		if ((actionList == null) || (actionList.length == 0))
		{
			throw new ConfigurationException("No actions in list");
		}

		final ArrayList<ActionPipelineProcessor> processorList = new ArrayList<ActionPipelineProcessor>();

		serviceMessageCounter = new ServiceMessageCounter(config);
		serviceMessageCounter.registerMBean();
		
		for (final ConfigTree actionConfig : actionList)
		{
			final String actionClassTag = actionConfig
 					.getAttribute(ListenerTagNames.ACTION_CLASS_TAG);
			if (LOGGER.isDebugEnabled())
			{
				LOGGER.debug("Registering action class " + actionClassTag);
			}
			final Class<?> actionClass;
			try
			{
				actionClass = ClassUtil.forName(actionClassTag, getClass());
			}
			catch (final ClassNotFoundException cnfe)
			{
				throw new ConfigurationException("Could not load action class "
						+ actionClassTag);
			}

			final ActionPipelineProcessor processor;
			if (BeanConfiguredAction.class.isAssignableFrom(actionClass))
			{
				if (LOGGER.isDebugEnabled())
				{
					LOGGER.debug("Using bean configured action processor for "
							+ actionClassTag);
				}
				processor = new BeanConfigActionProcessor(actionConfig,
						actionClass);
			}
			else if (ActionPipelineProcessor.class
					.isAssignableFrom(actionClass))
			{
				final ActionPipelineProcessor currentProcessor = (ActionPipelineProcessor) ActionProcessorMethodInfo
						.getActionClassInstance(actionConfig, actionClass);
				if (ActionProcessorMethodInfo.checkOverridden(actionConfig))
				{
					if (LOGGER.isDebugEnabled())
					{
						LOGGER
								.debug("Using overridden action pipeline processor for "
										+ actionClassTag);
					}
					processor = new OverriddenActionPipelineProcessor(
							actionConfig, currentProcessor);
				}
				else
				{
					if (LOGGER.isDebugEnabled())
					{
						LOGGER
								.debug("Using normal action pipeline processor for "
										+ actionClassTag);
					}
					processor = currentProcessor;
				}
			}
			else if (ActionLifecycle.class.isAssignableFrom(actionClass))
			{
				if (LOGGER.isDebugEnabled())
				{
					LOGGER
							.debug("Using overridden action lifecycle processor for "
									+ actionClassTag);
				}
				final ActionLifecycle currentLifecycle = (ActionLifecycle) ActionProcessorMethodInfo
						.getActionClassInstance(actionConfig, actionClass);
				processor = new OverriddenActionLifecycleProcessor(
						actionConfig, currentLifecycle);
			}
			else
			{
				LOGGER.warn("Action class " + actionClassTag
						+ " does not implement the ActionLifecycle interface");
				if (LOGGER.isDebugEnabled())
				{
					LOGGER.debug("Using overridden actions processor for "
							+ actionClassTag);
				}
				processor = new OverriddenActionProcessor(actionConfig,
						actionClass);
			}
			processorList.add(processor);
		}
		processors = processorList
				.toArray(new ActionPipelineProcessor[processorList.size()]);
	}

	/**
	 * Handle the initialisation of the pipeline
	 * 
	 * @throws ConfigurationException
	 *             For errors during initialisation.
	 */
	public void initialise() throws ConfigurationException
	{
		final int numLifecycles = processors.length;
		for (int count = 0; count < numLifecycles; count++)
		{
			final ActionLifecycle lifecycle = processors[count];
			try
			{
				lifecycle.initialise();
			}
			catch (final Exception ex)
			{
				handleDestroy(count - 1);
				throw new ConfigurationException(
						"Unexpected exception during lifecycle initialisation",
						ex);
			}
		}
		active.set(true);
	}

	/**
	 * Handle the destruction of the pipeline
	 */
	public void destroy()
	{
		active.set(false);
		handleDestroy(processors.length - 1);
	}

	/**
	 * Process the specified message.
	 * 
	 * @param message
	 *            The current message.
	 * @return true if the processing was successful, false otherwise.
	 */
	public boolean process(final Message message)
	{
		long start = System.nanoTime();
		serviceMessageCounter.incrementTotalCount();
		final Call callDetails = new Call() ;
		callDetails.copy(message.getHeader().getCall()) ;

		if (active.get())
		{
			if (LOGGER.isDebugEnabled())
			{
				LOGGER.debug("pipeline process for message: "+message.getHeader());
			}

			final int numProcessors = processors.length;
			final Message[] messages = new Message[numProcessors];

			Message currentMessage = message;

			for (int count = 0; count < numProcessors; count++)
			{
				final ActionPipelineProcessor processor = processors[count];
				messages[count] = currentMessage;

				try
				{
					LOGGER.debug("executing processor " + count+ " "+processor+" "+message.getHeader());
					
					currentMessage = processor.process(currentMessage);

					if (currentMessage == null)
					{
						break;
					}
				}
				catch (final Exception ex)
				{
					LOGGER
							.warn(
									"Unexpected exception caught while processing the action pipeline: "+message.getHeader(),
									ex);

					notifyException(count, ex, messages);

					/*
					 * Is this an application specific error? If so, try to return
					 * the error message to the identified recipient.
					 */
					
					final boolean throwRuntime = transactional && (ex instanceof RuntimeException) ;
					
					if (ex instanceof ActionProcessingFaultException)
					{
						ActionProcessingFaultException fault = (ActionProcessingFaultException) ex;

						if (fault.getFaultMessage() == null)
						{
							faultTo(callDetails, Factory.createErrorMessage(Factory.PROCESSING_ERROR, message, ex));
						}
						else
							faultTo(callDetails, fault.getFaultMessage());
					}
					else if (!throwRuntime)
					{
						faultTo(callDetails, Factory.createErrorMessage(Factory.UNEXPECTED_ERROR, message, ex));
					}

					long procTime = System.nanoTime() - start;
					serviceMessageCounter.update(new ActionStatusBean(procTime, count,
							ActionStatusBean.ACTION_FAILED));
		        	DeliveryObservableLogger.getInstance().logMessage(new MessageStatusBean(procTime, message, 
		        			MessageStatusBean.MESSAGE_FAILED));
					
		        	        if (throwRuntime)
		        	        {
		        	            throw (RuntimeException)ex ;
		        	        }
					return false;
				}
				serviceMessageCounter.update(new ActionStatusBean((System.nanoTime() - start), count,
						ActionStatusBean.ACTION_SENT));
			}

			// Reply...
			if (!oneWay)
			{
                            if (currentMessage != null)
                            {
                                replyTo(callDetails, currentMessage);
                            }
                            else if (!defaultProcessing)
                            {
                                LOGGER.warn("No response message for RequestResponse mep! " + callDetails);
                            }
			}

			notifySuccess(messages);
			long procTime = System.nanoTime() - start;
        	DeliveryObservableLogger.getInstance().logMessage(new MessageStatusBean(procTime, message, 
        			MessageStatusBean.MESSAGE_SENT));
			return true;
		}
		else
		{
			LOGGER.debug("pipeline process disabled for message: "+message.getHeader());

			faultTo(callDetails, Factory.createErrorMessage(Factory.NOT_ENABLED, message, null));
			long procTime = System.nanoTime() - start;
        	DeliveryObservableLogger.getInstance().logMessage(new MessageStatusBean(procTime, message, 
        			MessageStatusBean.MESSAGE_FAILED));
        	
			return false;
		}
	}
	
	/**
	 * Set the transactional flag for this pipeline.
	 * @param transactional true if running within a transaction, false otherwise.
	 */
	public void setTransactional(final boolean transactional)
	{
	    this.transactional = transactional ;
	}
	
        /**
         * Get the transactional flag for this pipeline.
         * @return true if running within a transaction, false otherwise.
         */
	public boolean isTransactional()
	{
	    return transactional ;
	}

	/**
	 * Send the reply.
	 * 
	 * @param callDetails
	 *            the call details for the original request.
	 * @param message
	 *            the message.
	 */

	private void replyTo(final Call callDetails, final Message message)
	{
		if (!DefaultReplyTo.initialiseReply(message, callDetails))
		{
		    if (defaultProcessing)
		    {
			LOGGER.warn("No reply to address defined for reply message! " + callDetails);
			sendToDLQ(callDetails, message, MessageType.reply) ;
		    }
		}
		else
		{
			final EPR replyToEPR = message.getHeader().getCall().getTo() ;
			messageTo(replyToEPR, message, MessageType.reply);
		}
	}

	/**
	 * Send the fault message to the EPR.
	 * 
	 * @param callDetails
	 *            the call details for the original request.
	 * @param faultToAddress
	 *            the EPR to target if one is not set in the message.
	 * @param message
	 *            the message.
	 */

	private void faultTo(final Call callDetails, final Message message)
	{
		if (!DefaultFaultTo.initialiseReply(message, callDetails, oneWay))
		{
		    if (defaultProcessing || oneWay)
		    {
			LOGGER.warn("No fault address defined for fault message! " + callDetails);
			sendToDLQ(callDetails, message, MessageType.fault) ;
		    }
		}
		else
		{
			final EPR faultToEPR = message.getHeader().getCall().getTo() ;
			messageTo(faultToEPR, message, MessageType.fault);
		}
	}
	
	/**
	 * Sent the message to the DLQ service.
	 * @param callDetails The original call details.
	 * @param message The response message.
	 * @param messageType The response type.
	 */
	private void sendToDLQ(final Call callDetails, final Message message,
		final MessageType messageType)
	{
		final Properties properties = message.getProperties() ;
		properties.setProperty(MessageStore.CLASSIFICATION, MessageStore.CLASSIFICATION_DLQ);
		properties.setProperty(ActionProcessingConstants.PROPERTY_FAILURE_CALL_DETAILS, callDetails.toString()) ;
		properties.setProperty(ActionProcessingConstants.PROPERTY_FAILURE_RESPONSE_TYPE, messageType.name()) ;

		try
		{
			final ServiceInvoker serviceInvoker = new ServiceInvoker(ServiceInvoker.dlqService) ;
			
			serviceInvoker.deliverAsync(message) ;
		}
		catch (final MessageDeliverException mde)
		{
			LOGGER.warn("Failed to send response failure to DLQ service") ;
			LOGGER.debug("Failed to send response failure to DLQ service", mde) ;
		}
	}

    private static enum MessageType {
        reply,
        fault,
    }

    private void messageTo(EPR epr, Message message, MessageType messageType) {
        if(epr instanceof LogicalEPR) {
            try {
                ServiceInvoker invoker = ((LogicalEPR)epr).getServiceInvoker();
                invoker.deliverAsync(message);
            } catch (MessageDeliverException e) {
                LOGGER.error("Failed to send " + messageType + " to address " + epr
                        + " for message "+message.getHeader(), e);
            }
        } else {
            Courier courier = null;

            try {
                courier = CourierFactory.getCourier(epr);
                courier.deliver(message);
            } catch (final CourierException e) {
                LOGGER.error("Failed to send " + messageType + " to address " + epr
                        + " for message " + message.getHeader(), e);
            } catch (final MalformedEPRException e) {
                LOGGER.error("Failed to send " + messageType + " to address " + epr
                        + " for message " + message.getHeader(), e);
            } catch (final Throwable e) {
                LOGGER.error("Failed to send " + messageType + " to address " + epr
                        + " for message " + message.getHeader(), e);
            } finally {
                if (courier != null) {
                    CourierUtil.cleanCourier(courier);
                }
            }
        }
    }

	/**
	 * Handle the destruction of the pipeline from the specified position.
	 * 
	 * @param initialPosition
	 *            The initial position to begin destruction.
	 */
	private void handleDestroy(final int initialPosition)
	{
		for (int count = initialPosition; count >= 0; count--)
		{
			final ActionLifecycle lifecycle = processors[count];
			try
			{
				lifecycle.destroy();
			}
			catch (final Exception ex)
			{
				LOGGER
						.warn(
								"Unexpected exception during lifecycle destruction",
								ex);
			}
		}
	}

	/**
	 * Notify the processors of an error during processing.
	 * 
	 * @param initialPosition
	 *            The position of the first processor to notify.
	 * @param ex
	 *            The exception which caused the failure.
	 * @param messages
	 *            The messages associated with successful processors.
	 */
	private void notifyException(final int initialPosition, final Exception ex,
			final Message[] messages)
	{
		for (int count = initialPosition; count >= 0; count--)
		{
			final ActionPipelineProcessor processor = processors[count];
			try
			{
				processor.processException(messages[count], ex);
			}
			catch (final Exception ex2)
			{
				LOGGER
						.warn(
								"Unexpected exception notifying processor of pipeline failure",
								ex2);
			}
		}
	}

	/**
	 * Notify the processors of a successful pipeline process.
	 * 
	 * @param messages
	 *            The messages associated with the processors.
	 */
	private void notifySuccess(final Message[] messages)
	{
		for (int count = messages.length - 1; count >= 0; count--)
		{
			final Message message = messages[count];
			if (message != null)
			{
				final ActionPipelineProcessor processor = processors[count];
				try
				{
					processor.processSuccess(messages[count]);
				}
				catch (final Exception ex)
				{
					LOGGER
							.warn(
									"Unexpected exception notifying processor of pipeline success",
									ex);
				}
			}
		}
	}
	
}
