/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.gateway.remotestrategies;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.jboss.cache.CacheException;
import org.jboss.soa.esb.listeners.gateway.remotestrategies.cache.FtpFileCache;
import org.jboss.soa.esb.listeners.gateway.remotestrategies.cache.FtpFileCacheException;

/**
 * Read-only implementation of {@link RemoteFileSystemStrategy}.
 * <p> 
 * An instance of this class can be used when a remote file system does
 * not support write operations.
 * </p>
 *  
 * @author <a href="daniel.bevenius@redpill.se">Daniel Bevenius</a>				
 *
 */
public class ReadOnlyRemoteFileSystemStrategy implements RemoteFileSystemStrategy
{
	private Logger log = Logger .getLogger( ReadOnlyRemoteFileSystemStrategy.class );
	
	private FtpFileCache ftpFileCache;
	
	/**
	 * Configures and starts the underlying FtpFileCache
	 */
	public void init( String configFile ) throws RemoteFileSystemStrategyException
	{
		try
		{
			ftpFileCache = new FtpFileCache( configFile );
		
			ftpFileCache.start();
		}
		catch ( FtpFileCacheException e )
		{
			throw new RemoteFileSystemStrategyException ( e );
		}
	}

	/**
	 * Do nothing
	 * 
	 * @param file		the file to delete. Ignored
	 * @return true	always returns true without deleting the filee
	 */
	public boolean deleteFile( final File file ) throws RemoteFileSystemStrategyException
	{
		return true;
	}

	/**
	 * Do nothing
	 * 
	 * @param from		the file to be renamed. Ignored
	 * @param to		the new file. Ignored
	 * @return true	always returns true without deleting the filee
	 */
	public boolean renameFile( final File from, final File to ) throws RemoteFileSystemStrategyException
	{
		return true;
	}

	/**
	 * This method simply returns the file without a worksuffix. 
	 * This is useful when the remote system does not allow one to rename the file.
	 * 
	 * @param file		the file to work with
	 * @param suffix	the suffix. This will be ignored
	 * @return File	the file untouched
	 */
	public File getWorkFileName( final File file, final String suffix )
	{
		return file;
	}

	/**
	 * Returns a File[] that only contains files that have not previously 
	 * been through this method.
	 */
	public File[] filterFileList( File[] files ) throws RemoteFileSystemStrategyException
	{
		if ( files == null )
			return files;
		
		List<File> newFileList = new ArrayList<File>();
		for ( File file : files )
		{
			String fileName = file.getName();
			if ( !ftpFileCache.containsFile( fileName ) )
			{
				try
				{
					ftpFileCache.putFileName( fileName );
					newFileList.add( file );
				} 
				catch (CacheException e)
				{
					log.error( "Exception while trying to put file : " + fileName + " into the cache", e);
				}
			}
		}
		
		return newFileList.toArray( new File[]{} );
	}
	
	public void removeAllData() throws CacheException
	{
		ftpFileCache.removeAll();
	}

	public void stop()
	{
		ftpFileCache.stop();
	}
	
	public void destroy()
	{
		ftpFileCache.destroy();
	}

	public void setDeleteOnEviction()
	{
		ftpFileCache.setDeleteOnEviction();
	}
	
	public Map getCacheListeners()
	{
		return ftpFileCache.getCacheListeners();
	}

	public void setCacheListener( Class cacheListenerClass ) throws RemoteFileSystemStrategyException 
	{
		try
		{
			ftpFileCache.setCacheListener( cacheListenerClass );
		} 
		catch (FtpFileCacheException e)
		{
			throw new RemoteFileSystemStrategyException( e );
		}
	}

}
