/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners.gateway;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.util.FileUtil;
import org.jboss.soa.esb.util.Util;

public class FileGatewayListener extends AbstractFileGateway {
    /**
     * serial version uid for this class
     */
    private static final long serialVersionUID = 8457176329093772449L;

    public FileGatewayListener(ConfigTree config) throws ConfigurationException, RegistryException, GatewayException {
        super(config);

        /*
           * JBESB-454 allowed gateways to pull files with any extension. Obvious
           * security issue, but we are explicit about this in the docs and users
           * should beware.
           */

        if ((_inputSuffix == null) || (_inputSuffix.equals(""))) {
            /*
                * If no suffix, then inputDir must be different from outputDir
                * or we go into an infinite loop. Already checked through
                * checkMyParams.
                */

            _inputFileFilter = null;

            /*
                * If no input suffix, then we must have a work suffix and we
                * must ignore worker files when sourcing new input files, or
                * we end up in an infinite loop.
                */

            _ignoreFileFilter = new IgnoreFile(_workingSuffix, _errorSuffix, _postProcessSuffix);
        } else
            _inputFileFilter = new FileEndsWith(_inputSuffix);
    }

    @Override
    protected void seeIfOkToWorkOnDir(File p_oDir) throws ConfigurationException {
        if (!p_oDir.exists())
            throw new ConfigurationException("Directory " + p_oDir.toString() + " not found");
        if (!p_oDir.isDirectory())
            throw new ConfigurationException(p_oDir.toString() + " is not a directory");
        if (!p_oDir.canRead())
            throw new ConfigurationException("Can't read directory " + p_oDir.toString());
        if (!p_oDir.canWrite())
            throw new ConfigurationException("Can't write/rename in directory " + p_oDir.toString());
    } // ________________________________

    @Override
    boolean deleteFile(File file) throws GatewayException {
        return file.delete();
    }

    @Override
    byte[] getFileContents(File file) throws GatewayException {
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            byte[] ba = new byte[1000];  // TODO magic number
            int iQread;
            FileInputStream inp = new FileInputStream(file);
            while (-1 != (iQread = inp.read(ba)))
                if (iQread > 0)
                    out.write(ba, 0, iQread);
            inp.close();
            out.close();
            return out.toByteArray();
        }
        catch (IOException e) {
            throw new GatewayException(e);
        }
    }

    @Override
    File[] getFileList() throws GatewayException {
        /*
           * JBESB-454 allows no input suffix. This means that we need to
           * ignore any worker files, which are written to the same directory,
           * but which have a well defined suffix. All other files (errors and
           * post processed) go into separate directories, so we don't need
           * to be concerned about them.
           */

        if (_inputFileFilter != null) {
            /*
            * Input suffix is not null.
            */

            return filterFiles(_inputDirectory, _inputFileFilter);
        } else {
            /*
            * Input suffix is null so ignore any worker files.
            */

            return filterFiles(_inputDirectory, _ignoreFileFilter);
        }
    }

    private File[] filterFiles(File dir, FileFilter filter) {
        File[] files = dir.listFiles(filter);

        if(files == null) {
            _logger.warn("Error applying file filter '" + filter + "' to directory '" + dir.getAbsolutePath() + "'.  Null file list returned.  Invalid directory, or IO error reading directory.");
        }

        return files;
    }

    @Override
    boolean renameFile(File from, File to) throws GatewayException {
        if (to.exists() && !to.delete()) {
            throw new GatewayException("Cannot delete target file: "
                    + to.getAbsolutePath());
        }

        return FileUtil.renameTo(from, to);
    }

    @Override
    String getDefaultComposer() {
         return LocalFileMessageComposer.class.getName();
    }

    @Override
    void bytesToFile(byte[] bytes, File file) throws GatewayException {
        try {
            if (file.exists() && !file.delete())
                throw new GatewayException("Unable to delete existing file " + file);
            FileOutputStream out = new FileOutputStream(file);
            out.write(bytes);
            out.close();
        }
        catch (Exception e) {
            throw new GatewayException(e);
        }
    }

    /**
     * This implementation allows no input suffix, whereas the base class
     * requires an input suffix.
     */

    protected void checkInputSuffix() throws ConfigurationException {
        // any suffix is allowed for input; set value to "" if null.

        if (_inputSuffix == null)
            _inputSuffix = "";
    }

    /**
     * Simple file filter for local filesystem Will accept only files that end
     * with the String supplied at constructor time
     */
    private class FileEndsWith implements FileFilter {
        String m_sSuffix;

        FileEndsWith(String p_sEnd) throws ConfigurationException {
            m_sSuffix = p_sEnd;
            if (Util.isNullString(m_sSuffix))
                throw new ConfigurationException("Must specify file extension");
        } // ______________________________

        public boolean accept(File p_f) {
            return (p_f.isFile()) ? p_f.toString().endsWith(m_sSuffix) : false;
        }
    }

    private class IgnoreFile implements FileFilter {
        String _workSuffix;
        String _errorSuffix;
        String _postSuffix;

        IgnoreFile(String workSuffix, String errorSuffix, String postSuffix) throws ConfigurationException {
            _workSuffix = workSuffix;
            _errorSuffix = errorSuffix;
            _postSuffix = postSuffix;

            if (Util.isNullString(_workSuffix))
                throw new ConfigurationException("Must specify workSuffix");
            if (Util.isNullString(_errorSuffix))
                throw new ConfigurationException("Must specify errorSuffix");
            if (Util.isNullString(_postSuffix))
                throw new ConfigurationException("Must specify postProcessSuffix");
        } // ______________________________

        public boolean accept(File p_f) {
            if (p_f.isFile()) {
                /*
                 * If file is a work file then ignore it.
                 */

                String fileName = p_f.toString();

                if (fileName.endsWith(_workSuffix) || fileName.endsWith(_errorSuffix) || fileName.endsWith(_postSuffix)) {
                    return false;
                } else
                    return true;
            } else
                return false;
        }
    }

    private FileFilter _inputFileFilter;  // normal file filter
    private FileFilter _ignoreFileFilter; // worker file filter (used if input suffix is null)


}
