/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config.mappers;

import java.util.List;

import org.apache.xmlbeans.XmlCursor;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.XmlCursor.TokenType;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.config.xbeanmodel.Bus;
import org.jboss.soa.esb.listeners.config.xbeanmodel.Listener;
import org.jboss.soa.esb.listeners.config.xbeanmodel.Provider;
import org.jboss.soa.esb.listeners.config.Generator.XMLBeansModel;
import org.jboss.soa.esb.listeners.config.xbeanmodel.PropertyDocument.Property;
import org.jboss.soa.esb.listeners.config.xbeanmodel.ServiceDocument.Service;
import org.jboss.soa.esb.listeners.message.MessageAwareListener;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Mapper utility methods.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
abstract class MapperUtil {

	/**
	 * Map all default listener attributes onto the target listener ConfigTree.
	 * @param listener The listener configuration instance.
	 * @param target The target DOM ConfigTree listener element.
	 * @param model The model.
	 */
	protected static void mapDefaultAttributes(Listener listener, Element target, XMLBeansModel model) {
		Service service = model.getService(listener);
		
		// Map maxThreads - appears to be a JMS only attrib...
		target.setAttribute(ListenerTagNames.MAX_THREADS_TAG, String.valueOf(listener.getMaxThreads()));
		target.setAttribute(ListenerTagNames.SERVICE_DESCRIPTION_TAG, service.getDescription());	
		if(listener.getIsGateway()) {
			target.setAttribute(ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG, service.getCategory());			
			target.setAttribute(ListenerTagNames.TARGET_SERVICE_NAME_TAG, service.getName());			
		} else {
            target.setAttribute(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG, service.getCategory());         
            target.setAttribute(ListenerTagNames.SERVICE_NAME_TAG, service.getName());
            if(!target.hasAttribute(ListenerTagNames.LISTENER_CLASS_TAG)) {
                target.setAttribute(ListenerTagNames.LISTENER_CLASS_TAG, MessageAwareListener.class.getName());
            }
        }
	}
	
	/**
	 * Map all the properties targeted at the supplied listener onto the target element.
	 * <p/>
	 * This involves copying the properties from the associated bus and provider nodes.
	 * @param listener The listener config instance.
	 * @param target The target DOM ConfigTree listener element.
	 * @param model The model.
	 * @throws ConfigurationException Bad bis configuration. 
	 */
	protected static void mapEPRProperties(Listener listener, Element target, XMLBeansModel model) throws ConfigurationException {
		Bus bus = model.getBus(listener.getBusidref());
		Provider provider = model.getProvider(bus);
		
		// Map the properties from the provider config...
		mapProperties(provider.getPropertyList(), target);
		// Map the properties from the bus config...
		mapProperties(bus.getPropertyList(), target);
	}
	
	/**
	 * Map all the supplied properties onto the target element.
	 * @param target The target DOM element.
	 * @param properties The properties to be mapped.
	 */
	protected static void mapProperties(List<Property> properties, Element target) {
		// Map the property elements to actions attributes...
		for(Property property : properties) {
         Element propertyElement = target.getOwnerDocument().createElement("property");
         serialize(property, propertyElement);
         if (propertyElement.hasChildNodes())
         {
            YADOMUtil.copyChildNodes(propertyElement, target);
         }
         else
         {
            target.setAttribute(property.getName(), property.getValue());
         }
      }
	}

	/**
	 * Serialise the supplied {@link XmlObject} instance to the supplied element.
	 * @param xmlObject The XmlObject instance.
	 * @param toElement The element to which the cursor is to be serialised.
	 */
	protected static void serialize(XmlObject xmlObject, Element toElement) {
		XmlCursor cursor = xmlObject.newCursor();
		
		// Note there are methods on the XmlObject that "looked" as though they might be able
		// to do this for us (save, newDomNode etc), but they kept throwing exceptions.
		
		serialize(cursor, toElement);
		cursor.dispose();
	}

	/**
	 * Serialise the XML content behind the supplied XmlCursor instance to the supplied
	 * target element.
	 * @param cursor The cursor instance.
	 * @param toElement The target DOM Element.
	 */
	private static void serialize(XmlCursor cursor, Element toElement) {
		TokenType token;
		Document doc = toElement.getOwnerDocument();

		while(true) {
			token = cursor.toNextToken();
			
			switch (token.intValue()) {
			case TokenType.INT_ATTR:
				toElement.setAttribute(cursor.getName().getLocalPart(), cursor.getTextValue());
				break;
			case TokenType.INT_COMMENT:
				toElement.appendChild(doc.createComment(cursor.getTextValue()));
				break;
			case TokenType.INT_START:
				Element childElement = doc.createElement(cursor.getName().getLocalPart());
				toElement.appendChild(childElement);
				serialize(cursor, childElement);
				break;
			case TokenType.INT_TEXT:
				toElement.appendChild(doc.createTextNode(cursor.getChars()));
				break;
			case TokenType.INT_END:
			case TokenType.INT_STARTDOC:
			case TokenType.INT_ENDDOC:
			case TokenType.INT_NONE:
				return;
			case TokenType.INT_PROCINST:
			case TokenType.INT_NAMESPACE:
			default:
				break;
			}
		}
	}
}
