/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config.mappers;

import java.util.HashMap;
import java.util.List;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.config.xbeanmodel.MepType;
import org.jboss.soa.esb.listeners.config.xbeanmodel.ActionDocument.Action;
import org.jboss.soa.esb.listeners.config.xbeanmodel.ActionsDocument.Actions;
import org.jboss.soa.esb.listeners.config.Generator.XMLBeansModel;
import org.jboss.soa.esb.listeners.config.xbeanmodel.PropertyDocument.Property;
import org.jboss.soa.esb.listeners.config.xbeanmodel.ServiceDocument.Service;
import org.w3c.dom.Element;

/**
 * Performs the mapping of XSD based configuration service actions onto a "ConfigTree"
 * style listener configuration.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class ActionMapper {
	private static final Logger logger = Logger.getLogger(ActionMapper.class);

	/**
	 * Perform the mapping.    Checks to ensure that action names are unique, and throws a ConfigurationException if 
	 * a duplicate action name is registered.
	 * @param listenerConfigTree The listener ConfigTree to which the actions are to be added.
	 * @param listenerService The Service to which the listener is bound.  The Service defines the actions.
	 * @param model The configuration model from which the mapping is being performed.
	 * @throws ConfigurationException 
	 */
	public static void map(Element listenerConfigTree, Service listenerService, XMLBeansModel model) throws ConfigurationException {
		if (listenerService.getActions()!=null) {
		        final Actions actions = listenerService.getActions();
		        final MepType.Enum mep  = actions.getMep() ;
		        if (mep != null)
		        {
		            listenerConfigTree.setAttribute(ListenerTagNames.MEP_ATTRIBUTE_TAG, mep.toString()) ;
		        }
			List<Action> actionList = actions.getActionList();
			
			try {
				HashMap<String,String> hm = new HashMap<String,String>();
				for(Action action : actionList) {
					mapAction(listenerConfigTree, action);
					if (hm.containsKey(action.getName())) {
						throw new ConfigurationException("Tried to register a duplicate action name.  Service " 
								+ listenerService.getName() + " already contains an action named "
								+ action.getName() + ".    Action names must be unique per service.");
					} else {
						hm.put(action.getName(), action.getName());
					}
				}
			} catch (ConfigurationException ce) {
				logger.error("", ce);
			}
		}
	}

	/**
	 * Map an action instance onto the listener ConfigTree.
	 * @param listenerConfigTree The target listener ConfigTree.
	 * @param action The action instance.
	 */
	private static void mapAction(Element listenerConfigTree, Action action) {
		Element actionConfigTree = YADOMUtil.addElement(listenerConfigTree, "action");
		List<Property> properties = action.getPropertyList();
		
		actionConfigTree.setAttribute(ListenerTagNames.ACTION_ELEMENT_TAG, action.getName());
		actionConfigTree.setAttribute(ListenerTagNames.ACTION_CLASS_TAG, action.getClass1());
		actionConfigTree.setAttribute(ListenerTagNames.PROCESS_METHOD_TAG, action.getProcess());
		// The "okMethods" and "exceptionMethod" attributes are supported by adding them as child <property> elements.
		
		// Map the property elements to action attributes...
		for(Property property : properties) {
			Element propertyElement = listenerConfigTree.getOwnerDocument().createElement("property");
			
			MapperUtil.serialize(property, propertyElement);

            if(property.getValue() != null) {
                actionConfigTree.setAttribute(property.getName(), property.getValue());
            }
            // If the property has child content, add that directy to the action.  Of course it only really
			// makes sense to do this with one property!
			if(propertyElement.hasChildNodes()) {
				YADOMUtil.copyChildNodes(propertyElement, actionConfigTree);
			}
		}
		
		// Remove any empty attributes set on the action config...
		YADOMUtil.removeEmptyAttributes(actionConfigTree);
	}
}
