package org.jboss.soa.esb.listeners.config;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.publish.ActionContractPublisher;
import org.jboss.internal.soa.esb.publish.ContractPublisher;
import org.jboss.internal.soa.esb.publish.Publish;
import org.jboss.soa.esb.listeners.config.xbeanmodel.ActionDocument.Action;
import org.jboss.soa.esb.listeners.config.Generator.XMLBeansModel;
import org.jboss.soa.esb.listeners.config.xbeanmodel.ServiceDocument.Service;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleController;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * Service contract publisher.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class ServicePublisher {

    private String serviceName;
    private String category;
    private String description;
    private ContractPublisher contractPublisher;
    private static Map<ManagedLifecycleController, List<ServicePublisher>> servicePublishers = new LinkedHashMap<ManagedLifecycleController, List<ServicePublisher>>();

    /**
     * Public constructor.
     *
     * @param name     Service name.
     * @param category Service category.
     * @param contractPublisher Contract publisher implementation.
     */
    protected ServicePublisher(String name, String category, ContractPublisher contractPublisher) {
        AssertArgument.isNotNullAndNotEmpty(name, "name");
        AssertArgument.isNotNullAndNotEmpty(category, "category");

        this.serviceName = name;
        this.category = category;
        this.contractPublisher = contractPublisher;
    }

    /**
     * Get the service description.
     * @return Service description.
     */
    public String getDescription() {
        return description;
    }

    /**
     * Set the service description.
     * @param description Service description.
     */
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * Get the name of the Service.
     *
     * @return The service name.
     */
    public String getServiceName() {
        return serviceName;
    }

    /**
     * Get the Service category.
     *
     * @return The service category.
     */
    public String getCategory() {
        return category;
    }

    /**
     * Get the contract publisher for the Service associated with this publisher.
     *
     * @return The contract publisher.
     */
    public ContractPublisher getContractPublisher() {
        return contractPublisher;
    }

    /**
     * Add service publication from the suppplied configuration, keying them under the
     * controller that is managing these services.
     * <p/>
     * This is used later to publish as service list, their EPRs (from the reg),
     * links to their contract definitions, and the contract definitions
     * (e.g. wsdls) themselves.
     *
     * @param model The configuration model.
     */
    public static void addServicePublishers(ManagedLifecycleController controller, XMLBeansModel model) {
        List<Service> serviceConfigs = model.getServices();
        List<ServicePublisher> publishers = new ArrayList<ServicePublisher>();

        publishers.clear();
        for (Service service : serviceConfigs) {
            ContractPublisher contractPublisher = getConractPublisher(service);
            ServicePublisher servicePublisher = new ServicePublisher(service.getName(), service.getCategory(), contractPublisher);

            servicePublisher.setDescription(service.getDescription());
            publishers.add(servicePublisher);
        }

        servicePublishers.put(controller, publishers);
    }

    /**
     * Get the full list of publishers registered against all active {@link ManagedLifecycleController}
     * instances.
     * @return The full list of publishers.
     */
    public static List<ServicePublisher> getServicePublishers() {
        List<ServicePublisher> publishers = new ArrayList<ServicePublisher>();
        Collection<List<ServicePublisher>> allPublishers = servicePublishers.values();

        for(List<ServicePublisher> curPublisherList : allPublishers) {
            publishers.addAll(curPublisherList);
        }

        return publishers;
    }

    /**
     * Remove the service publications for the services under the control of the supplied controller.
     * @param controller Controller.
     */
    public static void removeServicePublishers(ManagedLifecycleController controller) {
        servicePublishers.remove(controller);
    }

    @SuppressWarnings("unchecked")
    private static ContractPublisher getConractPublisher(Service service) {
        if(service.getActions() == null || service.getActions().getActionList() == null) {
            return null;
        }

        for (Action action : service.getActions().getActionList()) {
            Class<Class> actionClass;

            try {
                actionClass = (Class<Class>) ClassUtil.forName(action.getClass1(), ServicePublisher.class);
            } catch (ClassNotFoundException e) {
                throw new RuntimeException("Failed to find action class '" + action.getClass1() + "'.", e);
            }

            Publish publishAnnotation = (Publish) actionClass.getAnnotation(Publish.class);
            if (publishAnnotation != null) {
                Class publisherClass;
                ActionContractPublisher publisher = null;

                publisherClass = publishAnnotation.value();
                try {
                    publisher = (ActionContractPublisher) publisherClass.newInstance();
                    publisher.setActionConfig(action);
                    return publisher;
                } catch (ClassCastException e) {
                    throw new RuntimeException("Action Contract Publisher class '" + publisherClass.getName() + "' must implement " + ActionContractPublisher.class.getName());
                } catch (Exception e) {
                    throw new RuntimeException("Failed to instantiate Contract Publisher '" + publisherClass.getName() + "'. Class must implement a public default constructor.", e);
                }
            }
        }

        // No publisher configured on any of the actions in the processing chain...
        return null;
    }
}
