/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlException;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.config.xbeanmodel.JbossesbDocument.Jbossesb;
import org.jboss.soa.esb.listeners.config.xbeanmodel.ServiceDocument.Service;
import org.jboss.soa.esb.listeners.config.xbeanmodel.ServicesDocument.Services;
import org.jboss.soa.esb.listeners.config.xbeanmodel.ScheduleProviderDocument.ScheduleProvider;
import org.jboss.soa.esb.listeners.config.xbeanmodel.*;
import org.w3c.dom.Document;

/**
 * ESB Configuration Generator.
 * <p/>
 * Generates a legacy/deprecated configuration fileset in accordance with the
 * {@link org.jboss.soa.esb.helpers.ConfigTree} approach to ESB configuration.  The
 * generator input should be a configuration that has been validated against the
 * JBoss ESB configuration schema (XMLSchema).
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class Generator {
	
	private static Logger logger = Logger.getLogger(Generator.class);
	
	/**
	 * Gateway Listeners config file.
	 */
	public static final String ESB_CONFIG_GATEWAY_XML_FILE = "jbossesb-gateway.xml";
	/**
	 * ESB Aware Listeners config file.
	 */
	public static final String ESB_CONFIG_XML_FILE = "jbossesb-listener.xml";
	/**
	 * Configuration model.
	 */
	private XMLBeansModel model;

   private OutputStream gatewayOutputStream;

   private OutputStream listenerOutputStream;


   public Generator(InputStream config, OutputStream listenerOutputStream, OutputStream gatewayOutputStream) throws ConfigurationException, IOException
   {
      this(config);
      this.gatewayOutputStream = gatewayOutputStream;
      this.listenerOutputStream = listenerOutputStream;
   }

   /**
	 * Public constructor.
	 * @param config The input configuration stream.
	 * @throws ConfigurationException Bad listener ESB configuration. 
	 * @throws IOException Unable to read the ESB listener configuration.
	 */
	public Generator(InputStream config) throws ConfigurationException, IOException {
		if(config == null) {
			throw new IllegalArgumentException("null 'config' arg in constructor call.");
		}
		
		// Parse the config into our internal model instance...
       model = parseConfig(config);
       assertOneScheduleProviderConfig(model);
   }

    private void assertOneScheduleProviderConfig(XMLBeansModel model) throws ConfigurationException {
        ProvidersDocument.Providers providersConfig = model.jbossesb.getProviders();

        if(providersConfig == null) {
            return;
        }

        List<Provider> providers = providersConfig.getProviderList();
        int numScheduleProviders = 0;

        for(Provider provider : providers) {
            if(provider instanceof ScheduleProvider) {
                numScheduleProviders++;
            }
        }

        if(numScheduleProviders > 1) {
            throw new ConfigurationException("Configuration contains " + numScheduleProviders + " <schedule-provider> configurations.  Only one of this provider type can exist per configuration.");
        }
    }

    /**
    * Parse the supplied config into a config model.
    * @param config The input configuration stream.
    * @return Config model.
    * @throws ConfigurationException Bad listener ESB configuration.
    * @throws IOException Unable to read the ESB listener configuration.
    */
   public static XMLBeansModel parseConfig(InputStream config) throws IOException, ConfigurationException {
       try {
           JbossesbDocument doc = JbossesbDocument.Factory.parse(config);

           // TODO: Get validation working from here - would save
           // doc.validate();

           return new XMLBeansModel(doc);
       } catch (XmlException e) {
           throw new ConfigurationException("Error while processing ESB Listener configuration stream.", e);
       }
   }

   /**
    * Generate the configuration set in the supplied output directory and store it in the member outputstreams.
    * @throws ConfigurationException Failed to generate configuration set.
    */
   public void generate() throws ConfigurationException {
      // Generate and serialise the configuration for the ESB Aware listeners...
      ESBAwareGenerator awareGenerator = new ESBAwareGenerator(model);
      Document awareConfig = awareGenerator.generate();

      YADOMUtil.serialize(awareConfig, listenerOutputStream);

      // Generate and serialise the configuration for the Gateway listeners...
      GatewayGenerator gatewayGenerator = new GatewayGenerator(model);
      Document gatewayConfig = gatewayGenerator.generate();

      YADOMUtil.serialize(gatewayConfig, gatewayOutputStream);
   }

	/**
	 * Generate the configuration set in the supplied output directory.
	 * <p/>
	 * The names of the generated files will be esb-config.xml (ESB Aware Listeners configuration)
	 * and esb-config-gateway.xml (Gateway Listeners configuration).
	 * 
	 * @param outdir The output directory to where the configuration set is to be generated.
	 * The directory will be created if it doesn't already exist.
	 * @throws ConfigurationException Failed to generate configuration set.
	 */
	public void generate(File outdir) throws ConfigurationException {
		if(outdir == null) {
			throw new IllegalArgumentException("null 'outdir' arg in method call.");
		}
		if(!outdir.exists()) {
			outdir.mkdirs();
			if(!outdir.exists()) {
				throw new IllegalArgumentException("Failed to create 'outdir' [" + outdir.getAbsolutePath() + "].");
			}
		}
		if(!outdir.isDirectory()) {
			throw new IllegalArgumentException("'outdir' exists, but is not a directory.");
		}

		// Generate and serialise the configuration for the ESB Aware listeners...
		ESBAwareGenerator awareGenerator = new ESBAwareGenerator(model);
		Document awareConfig = awareGenerator.generate();

		logger.info("Serializing ESB Aware Listener 'ConfigTree' configuration to [" + (new File(outdir, ESB_CONFIG_XML_FILE)).getAbsolutePath() + "].");
		YADOMUtil.serialize(awareConfig, outdir, ESB_CONFIG_XML_FILE);

		// Generate and serialise the configuration for the Gateway listeners...
		GatewayGenerator gatewayGenerator = new GatewayGenerator(model);
		Document gatewayConfig = gatewayGenerator.generate();
		
		logger.info("Serializing ESB Gateway 'ConfigTree' configuration to [" + (new File(outdir, ESB_CONFIG_GATEWAY_XML_FILE)).getAbsolutePath() + "].");
		YADOMUtil.serialize(gatewayConfig, outdir, ESB_CONFIG_GATEWAY_XML_FILE);
	}

    public static Properties toProperties(List<PropertyDocument.Property> configProperties) {
        Properties properties = new Properties();

        for(PropertyDocument.Property property : configProperties) {
            properties.setProperty(property.getName(), property.getValue());
        }

        return properties;
    }

	/**
	 * Get the model instance associate with the configuration used to create this Generator instance. 
	 * @return Returns the model.
	 */
	public XMLBeansModel getModel() {
		return model;
	}
	
	/**
	 * XMLBeans based model implementation.
	 * 
	 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
	 */
	public static class XMLBeansModel {

		/**
		 * XMLBeans config model instance.
		 */
		private Jbossesb jbossesb;
		
		/**
		 * Constructor.
		 * @param xmlBeansDoc XMLBeans config model.
		 */
		private XMLBeansModel(JbossesbDocument xmlBeansDoc) {
			this.jbossesb = xmlBeansDoc.getJbossesb();
		}

        /**
		 * Get the list of ESB Gateway Listeners from the configuration.
		 * @return The list of ESB Gateway Listeners from the configuration.
         * @throws ConfigurationException Bad configuration.
		 */
		public List<Listener> getGatewayListeners() throws ConfigurationException {
			return getListeners(true);
		}

		/**
		 * Get the list of ESB Aware Listeners from the configuration.
		 * @return The list of ESB Aware Listeners from the configuration.
         * @throws ConfigurationException Bad configuration.
		 */
		public List<Listener> getESBAwareListeners() throws ConfigurationException {
			return getListeners(false);
		}

        /**
		 * Get the Service to which the supplied {@link Listener} configuration instance is bound.
		 * @param listener The listener instance (Gateway or ESB Aware).
		 * @return The Service to which the 
		 */
		public Service getService(Listener listener) {
            List<Service> services = getServices();

            for(Service service : services) {
				for(Listener serviceListener : service.getListeners().getListenerList()) {
					if(serviceListener == listener) {
						return service;
					}
				}
			}
			
			throw new IllegalStateException("No Service instance found for the supplied Listener instance.  This should not be possible if the Listener instance was provided by this configuration.  Where has this Listener instance come from?");
		}

        /**
         * Get the Service list.
         * @return Service list.
         */
        public List<Service> getServices() {
            final Services services = jbossesb.getServices();
            if (services != null) {
                return jbossesb.getServices().getServiceList();
            } else {
                return Collections.emptyList();
            }
        }

        /**
		 * Get the &lt;bus&gt; configuration matching the supplied busid reference value.
		 * @param busid The required &lt;bus&gt; configuration reference value.
		 * @return The Bus configuration instance.
		 * @throws ConfigurationException Unknown busid reference value.
		 */
		public Bus getBus(String busid) throws ConfigurationException {
			List<Provider> providers = jbossesb.getProviders().getProviderList();
			
			for(Provider provider : providers) {
                if(provider instanceof BusProvider) {
                    List<Bus> buses = ((BusProvider)provider).getBusList();

                    for(Bus bus : buses) {
                        if(bus.getBusid().equals(busid)) {
                            return bus;
                        }
                    }
                }
            }
			
			throw new ConfigurationException("Invalid ESB Configuration: No <bus> configuration matching busid reference value [" + busid + "].");
		}

		/**
		 * Get the &lt;provider&gt; configuration containing the supplied Bus configuration instance.
		 * @param bus The Bus config instance whose Provider is being sought.
		 * @return The Provider configuration instance.
		 */
		public Provider getProvider(Bus bus) {
			List<Provider> providers = jbossesb.getProviders().getProviderList();
			
			for(Provider provider : providers) {
                if(provider instanceof BusProvider) {
                    List<Bus> buses = ((BusProvider)provider).getBusList();

                    for(Bus installedBus : buses) {
                        if(installedBus == bus) {
                            return provider;
                        }
                    }
                }
            }
			
			throw new IllegalStateException("No Provider instance found for the supplied Bus config instance.  This should not be possible if the Bus instance was provided by this configuration.  Where has this Bus instance come from?");
		}

        public ScheduleProvider getScheduleProvider() {
            if(jbossesb.getProviders() == null) {
                return null;
            }

            List<Provider> providers = jbossesb.getProviders().getProviderList();

            for(Provider provider : providers) {
                if(provider instanceof ScheduleProvider) {
                    return (ScheduleProvider) provider;
                }
            }

            return null;
        }

        private boolean isGateway(Listener listener)
      {
         return listener.getIsGateway() || (listener instanceof JcaGatewayDocument.JcaGateway);
      }

      /**
		 * Get the list of ESB Listeners based on their Gateway flag.
		 * @return The list of ESB Aware or Gateway Listeners from the configuration.
         * @param isGateway Is the listener a gateway or ESB aware listener.
         * @throws ConfigurationException Bad configuration.
		 */
		private List<Listener> getListeners(boolean isGateway) throws ConfigurationException {
			List<Listener> gateways = new ArrayList<Listener>();
			if (jbossesb.getServices() != null) {
                List<Service> services = getServices();

                for(Service service : services) {
                    boolean listenerAdded = false;

					for(Listener listener : service.getListeners().getListenerList()) {
						if(isGateway(listener) == isGateway) {
							gateways.add(listener);
                            listenerAdded = true;
                        }
					}

                    // Make sure each Service config has a message aware listener...
                    // http://jira.jboss.com/jira/browse/JBESB-648
                    if(!isGateway && !listenerAdded) {
                        throw new ConfigurationException("Service configuration for Service '" + service.getCategory() + ":" + service.getName() + "' doesn't define a Message-Aware Listener (i.e. is-gateway='false').");
                    }
                }
			}
			
			return gateways;
		}
		/**
		 * Gets the setting for the number of seconds between reloads.
		 * 
		 * @return The param reload seconds config value.
		 */
		public String getParameterReloadSecs() {
			return jbossesb.getParameterReloadSecs().getStringValue();
		}
        
        public static String getProperty(List<PropertyDocument.Property> properties, String name, String defaultVal) {
            for (PropertyDocument.Property property : properties) {
                if(property.getName().equals(name)) {
                    return property.getValue();
                }
            }

            return defaultVal;
        }
	}
}
