/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners;

import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.Initializable;
import org.jboss.soa.esb.schedule.SchedulingException;
import org.jboss.internal.soa.esb.assertion.AssertArgument;

/**
 * Scheduled Event Composer.
 *
 * @author <a href="daniel.bevenius@redpill.se">Daniel Bevenius</a>
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public interface ScheduledEventMessageComposer extends Initializable {

    /**
     * Compose an ESB "aware" message.
     * <p/>
     * Called by the {@link org.jboss.soa.esb.schedule.ScheduledEventListener#onSchedule()} method.
     *
     * @return ESB aware message instance. If this method returns null the action pipeline will not be processed.
     * @throws SchedulingException Failed to compose message on scheduled event.
     */
    public Message composeMessage() throws SchedulingException;

    /**
     * Called by the {@link org.jboss.soa.esb.schedule.ScheduledEventListener} after processing of the
     * {@link org.jboss.soa.esb.message.Message} has completed.
     * <p/>
     * This event can be used to perform any necessary cleanup processing for the {@link Message} instance etc.
     *
     * @param message ESB aware message instance to be decomposed.
     * @return ESB unaware message instance - a message payload.
     * @throws SchedulingException Error while processing the onProcessingComplete event.
     */
    public Message onProcessingComplete(Message message) throws SchedulingException;

    /**
     * Uitility factory class for reflective {@link ScheduledEventMessageComposer} construction.
     */
    public static class Factory {

        /**
         * Factory method.
         *
         * @param className Class name.
         * @return Composer instance.
         * @throws ConfigurationException Unable to construct composer.
         */
        public static ScheduledEventMessageComposer createInstance(String className) throws ConfigurationException {
            AssertArgument.isNotNullAndNotEmpty(className, "className");

            try {
                return (ScheduledEventMessageComposer) ClassUtil.forName(className, Factory.class).newInstance();
            } catch (ClassNotFoundException e) {
                throw new ConfigurationException("Composer class [" + className + "] not found in classpath.", e);
            } catch (InstantiationException e) {
                throw new ConfigurationException("Failed to instantiate composer class [" + className + "].", e);
            } catch (IllegalAccessException e) {
                throw new ConfigurationException("Failed to instantiate composer class [" + className + "].", e);
            }
        }
    }
}
