/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Properties;
import java.util.Set;

import javax.jms.JMSException;
import javax.naming.Context;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.FTPEpr;
import org.jboss.soa.esb.addressing.eprs.FileEpr;
import org.jboss.soa.esb.addressing.eprs.JDBCEpr;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.util.Util;

public class ListenerUtil
{
    private static Logger logger = Logger.getLogger(ListenerUtil.class);

    private ListenerUtil()
	{
	}

    /**
     * For unittest/quickstart use, to deliver a message. This method closes all
     * JmsCOnnectionPools.
     * 
     * @param message
     * @param category
     * @param name
     * @throws MessageDeliverException
     * @throws RegistryException
     * @throws JMSException
     * @deprecated <b>DO NOT USE</b>.  Use {@link org.jboss.soa.esb.client.ServiceInvoker#deliverAsync(org.jboss.soa.esb.message.Message)} instead.
     */
	public static void tryToDeliver(Message message, String category, String name)
			throws MessageDeliverException, JMSException, RegistryException
	{
        logger.warn("**** DEPRECATED ***: Please do not call '" + ListenerUtil.class.getSimpleName() + ".tryToDeliver'! Use '" + ServiceInvoker.class.getSimpleName() + ".deliverAsync' instead.");
        new ServiceInvoker(category, name).deliverAsync(message);
	}

	public static EPR assembleEpr(ConfigTree tree)
			throws ConfigurationException
	{
		String urlString = tree.getAttribute(ListenerTagNames.URL_TAG);
		String protocol = (null == urlString) ? tree
				.getAttribute(ListenerTagNames.PROTOCOL_TAG) : urlString
				.split(":")[0];

		try
		{
			if ("jms".equals(protocol))
				return jmsEprFromElement(tree);
			if ("file".equals(protocol))
				return fileEprFromElement(tree);
			if ("ftp".equals(protocol))
				return fileEprFromElement(tree);
			if ("jdbc".equals(protocol))
				return jdbcEprFromElement(tree);
		}
		catch (Exception e)
		{
			e.printStackTrace();
			
			_logger.error("Problem", e);
			throw new ConfigurationException(e);
		}

		throw new ConfigurationException("Unknown protocol <" + protocol + ">");

	} // ________________________________

	public static JMSEpr jmsEprFromElement(ConfigTree tree)
			throws ConfigurationException
	{
		try
		{

			String name = tree.getRequiredAttribute(JMSEpr.DESTINATION_NAME_TAG);
            Set<String> names=tree.getAttributeNames();
            Properties environment = new Properties();
            for (String attributeName : names) {
                if (attributeName.startsWith("java.naming.")) {
                    environment.setProperty(attributeName, tree.getAttribute(attributeName));
                }
            }
			String type = getAttrAndWarn(tree, JMSEpr.DESTINATION_TYPE_TAG,
					"queue");
			String jndiURL = tree.getAttribute(JMSEpr.JNDI_URL_TAG);
            if (jndiURL==null) jndiURL = Configuration.getJndiServerURL() ;
            environment.setProperty(Context.PROVIDER_URL, jndiURL);
			String jndiContextFactory = tree.getAttribute(JMSEpr.JNDI_CONTEXT_FACTORY_TAG);
            if (jndiContextFactory==null) jndiContextFactory = Configuration.getJndiServerContextFactory() ;
            environment.setProperty(Context.INITIAL_CONTEXT_FACTORY, jndiContextFactory);
			String jndiPkgPrefix = tree.getAttribute(JMSEpr.JNDI_PKG_PREFIX_TAG);
            if (jndiPkgPrefix==null ) jndiPkgPrefix = Configuration.getJndiServerPkgPrefix() ;
            environment.setProperty(Context.URL_PKG_PREFIXES, jndiPkgPrefix);
            
			String jmsFactoryClass = getAttrAndWarn(tree,
					JMSEpr.CONNECTION_FACTORY_TAG, "ConnectionFactory");

			String selector = tree.getAttribute(JMSEpr.MESSAGE_SELECTOR_TAG);
			if (Util.isNullString(selector))
				_logger.debug("No value specified for "
						+ JMSEpr.MESSAGE_SELECTOR_TAG + " attribute"
						+ " -  All messages in queue <" + name
						+ "> will be picked up by listener");

			final String username = tree.getAttribute(JMSEpr.JMS_SECURITY_PRINCIPAL_TAG);
			final String password = tree.getAttribute(JMSEpr.JMS_SECURITY_CREDENTIAL_TAG);
			JMSEpr epr = new JMSEpr(JMSEpr.ONE_ONE_PROTOCOL,type, name, jmsFactoryClass, environment, 
					selector, 
					Boolean.valueOf(tree.getAttribute(JMSEpr.PERSISTENT_TAG)),
					tree.getAttribute(JMSEpr.ACKNOWLEDGE_MODE_TAG),
					username,
					password,
					Boolean.valueOf(tree.getAttribute(JMSEpr.TRANSACTED_TAG)));
			return epr;
		}
		catch (Exception e)
		{
			throw new ConfigurationException(e);
		}
	} // ________________________________

	public static FileEpr fileEprFromElement(ConfigTree tree) throws ConfigurationException
	{
		try
		{
            String urlString = tree.getRequiredAttribute(ListenerTagNames.URL_TAG);
			URL url = new URL(urlString);
			String protocol = url.getProtocol();
	
			if ("file".equals(protocol))
			{
				if (!new File(url.getFile()).isDirectory())
					throw new ConfigurationException("Attribute "
							+ ListenerTagNames.URL_TAG
							+ " must reference a directory");
			}
	
			FileEpr epr = ("file".equals(protocol)) ? new FileEpr(url) : ("ftp"
					.equals(protocol)) ? new FTPEpr(url) : null;
			if (null == epr)
				throw new ConfigurationException("Unsupported file protocol : "
						+ protocol);
	
			String inputSuffix = tree.getAttribute(FileEpr.INPUT_SUFFIX_TAG);
	
			if (!Util.isNullString(inputSuffix))
				epr.setInputSuffix(inputSuffix);
	
			boolean bErrorDel = Boolean.parseBoolean(getAttrAndWarn(tree,
					FileEpr.ERROR_DEL_TAG, "true"));
			String errorDir = tree.getAttribute(FileEpr.ERROR_DIR_TAG);
			String errorSuffix = tree.getAttribute(FileEpr.ERROR_SUFFIX_TAG);
			
			if (bErrorDel)
			{
				if (null != errorDir || null != errorSuffix)
					_logger
							.warn("If you don't specify "
									+ FileEpr.ERROR_DEL_TAG
									+ "'false' ,"
									+ FileEpr.ERROR_DIR_TAG
									+ " and "
									+ FileEpr.ERROR_SUFFIX_TAG
									+ " will have no effect because files in error will be deleted");
			}
			if (null == errorDir)
			{
				errorDir = url.getFile();
				warnDefault(FileEpr.ERROR_DIR_TAG, errorDir);
			}
			if (null == errorSuffix)
			{
				errorSuffix = ".esbERROR";
				warnDefault(FileEpr.ERROR_SUFFIX_TAG, errorSuffix);
			}
			epr.setErrorDelete(bErrorDel);
			epr.setErrorDirectory(errorDir);
			epr.setErrorSuffix(errorSuffix);
	
			boolean bPostDel = Boolean.parseBoolean(getAttrAndWarn(tree,
					FileEpr.POST_DEL_TAG, "true"));
	
			String postDir = tree.getAttribute(FileEpr.POST_DIR_TAG);
			String postSuffix = tree.getAttribute(FileEpr.POST_SUFFIX_TAG);
			if (bPostDel)
			{
				if (null != postDir || null != postSuffix)
					_logger
							.warn("If you don't specify "
									+ FileEpr.POST_DEL_TAG
									+ "'false' ,"
									+ FileEpr.POST_DIR_TAG
									+ " and "
									+ FileEpr.POST_SUFFIX_TAG
									+ " will have no effect because processed input messages will be deleted");
			}
			if (null == postDir)
			{
				postDir = url.getFile();
				warnDefault(FileEpr.POST_DIR_TAG, postDir);
			}
			if (null == postSuffix)
			{
				postSuffix = ".esbDONE";
				warnDefault(FileEpr.POST_SUFFIX_TAG, postSuffix);
			}
			epr.setPostDelete(bPostDel);
			epr.setPostDirectory(postDir);
			epr.setPostSuffix(postSuffix);
	
			if (epr instanceof FTPEpr)
			{
				FTPEpr ftp = (FTPEpr) epr;
				ftp.setPassive(Boolean.valueOf(getAttrAndWarn(tree,
						FTPEpr.PASSIVE_TAG, "false")));
			}
			return epr;
		}
		catch (URISyntaxException ex)
		{
			throw new ConfigurationException(ex);
		}
		catch (MalformedURLException ex)
		{
			throw new ConfigurationException(ex);
		}
	} // ________________________________

	public static JDBCEpr jdbcEprFromElement(ConfigTree tree) throws ConfigurationException
	{
		String url = tree.getRequiredAttribute(JDBCEpr.URL_TAG);
		if (!url.toLowerCase().startsWith("jdbc"))
			throw new ConfigurationException("URL in "
					+ ListenerTagNames.URL_TAG + " must be a jdbc URL");

		try
		{
			boolean bPostDel = Boolean.valueOf(tree.getAttribute(
					JDBCEpr.POST_DEL_TAG, "true"));
			boolean bErrorDel = Boolean.valueOf(tree.getAttribute(
					JDBCEpr.ERROR_DEL_TAG, "true"));
			JDBCEpr epr = new JDBCEpr(url, bPostDel, bErrorDel);
			
			String datasource = tree.getAttribute(JDBCEpr.DATASOURCE_TAG);
			
			if (datasource == null)
			{
				epr.setDriver(tree.getRequiredAttribute(JDBCEpr.DRIVER_TAG));
				epr.setUserName(getAttrAndWarn(tree, JDBCEpr.USERNAME_TAG, ""));
				epr.setPassword(getAttrAndWarn(tree, JDBCEpr.PASSWORD_TAG, ""));
			}
			else
				epr.setDatasource(datasource);
			
			epr.setTableName(tree.getRequiredAttribute(JDBCEpr.TABLE_NAME_TAG));
			epr.setMessageIdColumn(getColName(tree, JDBCEpr.MESSAGE_ID_COLUMN_TAG));
			epr.setStatusColumn(getColName(tree, JDBCEpr.STATUS_COLUMN_TAG));
			epr.setDataColumn(getColName(tree, JDBCEpr.DATA_COLUMN_TAG));
			epr.setTimestampColumn(getColName(tree, JDBCEpr.TIMESTAMP_COLUMN_TAG));
	
			return epr;
		}
		catch (URISyntaxException ex)
		{
			throw new ConfigurationException(ex);
		}
	} // ________________________________

	private static final String s_Sfx = "_column";

	private static String getColName(ConfigTree tree, String tag)
			throws ConfigurationException
	{
		String defaultColname = (tag.endsWith(s_Sfx) ? tag.substring(0, tag
				.length()
				- s_Sfx.length()) : null);
		return getAttrAndWarn(tree, tag, defaultColname);
	} // ________________________________

	public static String getAttrAndWarn(ConfigTree tree, String tag,
			String defaultValue) throws ConfigurationException
	{
		String value = null;
		try
		{
			value = tree.getAttribute(tag);
			if (null == value)
			{
				if (null == defaultValue)
					throw new ConfigurationException("Missing or invalid "
							+ tag + " attribute");
				else
				{
					warnDefault(tag, defaultValue);
					value = defaultValue;
				}
			}
			return value;
		}
		catch (Exception e)
		{
			throw new ConfigurationException(e);
		}
	} // ________________________________

	/**
         * Find an attribute in the tree (arg 0) or assign default value (arg 2)
         * If no such value exists then the default value will be returned, no
         * matter what the value. This differs from obtainAtt, which will throw a
         * ConfigurationException if no value is present and the default value is
         * null.
         * 
         * @param p_oP
         *            ConfigTree - look for attributes in this Element only
         * @param p_sAtt
         *            String - Name of attribute to find
         * @param p_sDefault
         *            String -default value if requested attribute is not there
         * @return String - value of attribute, or default value (if null)
         */
	
        public static String getValue (ConfigTree p_oP, String p_sAtt, String p_sDefault)
        {
                String sVal = p_oP.getAttribute(p_sAtt);

                return (sVal != null) ? sVal : p_sDefault;
        } // ________________________________
        
        /**
         * Default value of null.
         * 
         * @param p_oP
         * @param p_sAtt
         * @return
         */
        
        public static String getValue (ConfigTree p_oP, String p_sAtt)
        {
              return getValue(p_oP, p_sAtt, null);
        } // ________________________________
        
        /**
         * Find an attribute in the tree (arg 0) or assign default value (arg 2).
         * If there is no value associated with the attribute and the default
         * value is null, then a ConfigurationException will be thrown. For more
         * intuitive behaviour in the presence of default values, use getValue.
         * 
         * @param p_oP
         *            ConfigTree - look for attributes in this Element only
         * @param p_sAtt
         *            String - Name of attribute to find
         * @param p_sDefault
         *            String -default value if requested attribute is not there
         * @return String - value of attribute, or default value (if null)
         * @throws ConfigurationException -
         *             If requested attribute not found and no default value
         *             supplied by invoker
         * @deprecated <b>DO NOT USE</b>.  Use {@link ListenerUtil#getValue()} instead.
         */
        
        public static String obtainAtt (ConfigTree p_oP, String p_sAtt, String p_sDefault)
                        throws ConfigurationException
        {
            String sVal = getValue(p_oP, p_sAtt, p_sDefault);
            
            if ((sVal == null) && (p_sDefault == null))
                        throw new ConfigurationException(
                                        "Missing or invalid <" + p_sAtt + "> attribute");

            return (sVal != null) ? sVal : p_sDefault;
        } // ________________________________

	private static final boolean LOGWARN = true;

	private static void warnDefault(String tag, String defaultValue)
	{
		if (LOGWARN)
			_logger.debug("No value specified for " + tag + " attribute"
					+ " -  Using default value: '" + defaultValue + "'");
	} // ________________________________
	
	public static long getMaxMillisGatewayWait(ConfigTree tree, Logger logr)
		throws ConfigurationException
	{
		long maxWait = -1;
		String sAux = tree.getAttribute(ListenerTagNames.GATEWAY_WAIT_MILLIS_TAG);
		if (!Util.isNullString(sAux))
		{
			try
			{
				maxWait = Long.parseLong(sAux);
				if (maxWait < 1)
					logr.info("Value specified for "
							+ ListenerTagNames.GATEWAY_WAIT_MILLIS_TAG
							+ " ("+maxWait
							+") implies that this will be an 'inbound-only' gateway");
			}
			catch (NumberFormatException e)
			{
				throw new ConfigurationException("Invalid value for "
						+ListenerTagNames.GATEWAY_WAIT_MILLIS_TAG,e);
			}
		}
		else
		{
			logr.info("No value specified for: "
					+ ListenerTagNames.GATEWAY_WAIT_MILLIS_TAG
					+ " -  This will be an 'inbound-only' gateway");
		}
		return maxWait;
	} //________________________________

	private static final Logger _logger = Logger.getLogger(ListenerUtil.class);
}
