/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.couriers;

import java.util.ArrayList;
import java.util.Collection;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.message.Message;
/**
 * Handle delivery of a message to a collection of Couriers.
 * 
 * @author kevin
 */
public class CourierCollection
{
    /**
     * The array of couriers.
     */
    private final Courier[] couriers ;
    
    /**
     * Construct the courier collection.
     * @param eprs The collection of courier eprs.
     */
    public CourierCollection(final Collection<EPR> eprs)
        throws MalformedEPRException, CourierException
    {
        if ((eprs != null) && (eprs.size() > 0))
        {
            final ArrayList<Courier> courierList = new ArrayList<Courier>() ;
            try
            {
                for (EPR epr: eprs)
                {
                    final Courier courier = CourierFactory.getCourier(epr) ;
                    if (courier == null)
                    {
                        throw new CourierException("Null courier returned for EPR: " + epr) ;
                    }
                    courierList.add(courier) ;
                }
                couriers = courierList.toArray(new Courier[courierList.size()]) ;
                courierList.clear() ;
            }
            finally
            {
                if (courierList.size() > 0)
                {
                    for(Courier courier: courierList)
                    {
                        CourierUtil.cleanCourier(courier) ;
                    }
                }
            }
        }
        else
        {
            couriers = null ;
        }
    }
    
    /**
     * Deliver the message to the list of couriers.
     * @param message The message to deliverAsync.
     * @throws CourierException For processing errors.
     * @throws MalformedEPRException For configuration errors.
     */
    public boolean deliver(final Message message)
        throws CourierException, MalformedEPRException
    {
        if (couriers != null)
        {
            for(Courier courier: couriers)
            {
                courier.deliver(message) ;
            }
            return true ;
        }
        else
        {
            return false ;
        }
    }
    
    /**
     * Cleanup and release all couriers.
     */
    public void cleanup()
    {
        for(Courier courier: couriers)
        {
            CourierUtil.cleanCourier(courier) ;
        }
    }
}
