package org.jboss.soa.esb.addressing.eprs;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/**
 * This class represents the endpoint reference for services.
 */

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.XMLUtil;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * A helper class for using FTP style EPRs. Simply create instances of this
 * class instead of the base EPR. Since URLs can use FTP, we try to leverage
 * that as much as possible.
 * 
 * @author marklittle
 * 
 */
public class SFTPEpr extends FTPEpr
{
	public static final String CERTIFICATE_TAG = "certificate";

	public SFTPEpr(EPR epr)
	{
		super(epr);
		
		if (epr instanceof SFTPEpr)
		    certificateSet = ((SFTPEpr) epr).certificateSet;
	}
	
	public SFTPEpr (EPR epr, Element header)
	{
		super(epr, header);
		
		NodeList nl = header.getChildNodes();

		for (int i = 0; i < nl.getLength(); i++)
		{
			String prefix = nl.item(i).getPrefix();
			String tag = nl.item(i).getLocalName();
			
			try
			{
				if ((prefix != null) && (prefix.equals(XMLUtil.JBOSSESB_PREFIX)))
				{
					if ((tag != null) && (tag.equals(CERTIFICATE_TAG)))
					{
						getAddr().addExtension(CERTIFICATE_TAG, nl.item(i).getTextContent());
						certificateSet = true;
					}
				}
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}

	public SFTPEpr(URL url) throws URISyntaxException
	{
		super(url);
	}

	public SFTPEpr(URL url, URL cert) throws URISyntaxException
	{
		super(url);
		
		setCertificateURL(cert);
	}

	public SFTPEpr(String url, String cert) throws URISyntaxException, MalformedURLException
	{
		super(url);	

		setCertificateURL(new URL(cert));
	}

	/**
	 * Set the Certificate URL for this endpoint.
	 * 
	 * @param cert
	 *            the address.
	 * @throws URISyntaxException
	 *             thrown if this EPR is malformed.
	 */

	public final void setCertificateURL(URL cert) throws URISyntaxException
	{
		if (cert == null)
			throw new IllegalArgumentException();

		if (certificateSet)
			throw new IllegalStateException("Certificate URL already set.");

		getAddr().addExtension(CERTIFICATE_TAG, cert.toString());
		certificateSet = true;
	}

	/**
	 * @return the certificate URL for this EPR.
	 * @throws URISyntaxException
	 *             thrown if this EPR is malformed.
	 */

	public final URL getCertificateURL () throws URISyntaxException,
			MalformedURLException
	{
		String cert = getAddr().getExtensionValue(CERTIFICATE_TAG);

		if (cert != null)
			return new URL(cert);
		else
			return null;
	}

	public EPR copy ()
	{
	    return new SFTPEpr(this);
	}
	
	public String toString ()
	{
		return "SFTPEpr [ "+super.getAddr().extendedToString()+" ]";
	}
	
	public static URI type ()
	{
		return _type;
	}

	private boolean certificateSet = false;

	private static URI _type;

	static
	{
		try
		{
			_type = new URI("urn:jboss/esb/epr/type/sftp");
		}
		catch (Exception ex)
		{
			ex.printStackTrace();

			throw new ExceptionInInitializerError(ex.toString());
		}
	}
}