/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.routing;

import java.io.Serializable;
import java.net.URISyntaxException;
import java.util.Properties;

import javax.jms.BytesMessage;
import javax.jms.DeliveryMode;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageProducer;
import javax.jms.ObjectMessage;
import javax.jms.Queue;
import javax.jms.TextMessage;
import javax.naming.Context;
import javax.naming.NamingException;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.rosetta.pooling.ConnectionException;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPool;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPoolContainer;
import org.jboss.internal.soa.esb.rosetta.pooling.JmsSession;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.helpers.NamingContextException;
import org.jboss.soa.esb.helpers.NamingContextPool;
import org.jboss.soa.esb.notification.jms.DefaultJMSPropertiesSetter;
import org.jboss.soa.esb.notification.jms.JMSPropertiesSetter;
import org.jboss.soa.esb.util.Util;

/**
 * JMS Routing Action Processor.
 * <p/>
 * Sample Action Configuration:
 * <pre>
 * &lt;action class="org.jboss.soa.esb.actions.routing.JMSRouter"&gt;
 *     jndiName="queue/A"
 *     message-prop-<i>&gt;prop-name&lt;</i>="<i>&gt;prop-value&lt;</i>" &gt;!-- (Optional)--&lt; 
 *     unwrap="true/false" &gt;!-- (Optional - default false)--&lt;
 *     persistent="true/false" &gt;!-- (Optional - default true)--&lt;
 *     priority="integer" &gt;!-- (Optional - default Message.DEFAULT_PRIORITY)--&lt;
 *     time-to-live="long" &gt;!-- (Optional - default Message.DEFAULT_TIME_TO_LIVE)--&lt;
 * /&gt;
 * </pre>
 * Note how properties to be set on the message are prefixed with "message-prop-".
 * <p/>
 * <ul>
 * <li>'unwrap' true will extract the message payload from the Message object before sending</li>
 * <li>'unwrap' false false will send the serialized Message object</li>
 * </ul>
 * 
 * TODO: Add support for JMS Topic destinations.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 * @author <a href="mailto:daniel.bevenius@redpill.se">daniel.bevenius@redpill.se</a>
 * @since Version 4.0
 */
public class JMSRouter extends AbstractRouter {
    /**
     * Logger.
     */
    private static Logger logger = Logger.getLogger(JMSRouter.class);
    /**
     * Constant used in configuration 
     */
    public static final String PERSISTENT_ATTR = "persistent";
    /**
     * Constant used in configuration
     */
    public static final String PRIORITY_ATTR = "priority";
    /**
     * Constant used in configuration
     */
    public static final String TIME_TO_LIVE_ATTR = "time-to-live";
    /**
     * Routing properties.
     */
    private ConfigTree properties;
    /**
     * The JMS Queue name from the configuration
     */
    private String queueName;
    /**
     * JMS Queue setup.
     */
    private JMSSendQueueSetup queueSetup;
    /**
     * Strategy for setting JMSProperties
     */
    private JMSPropertiesSetter jmsPropertiesStrategy = new DefaultJMSPropertiesSetter();
    /**
     * Whether messages sent by this router should be sent with delivery mode
     * DeliveryMode.PERSISTENT or DeliveryMode.NON_PERSISTENT
     * Default is to send messages persistently
     */
    private int deliveryMode = DeliveryMode.PERSISTENT;
    /**
     * The priority for messages sent with this router
     */
    private int priority = Message.DEFAULT_PRIORITY;
    /**
     * The time-to-live for messages sent with this router
     */
    private long timeToLive = Message.DEFAULT_TIME_TO_LIVE;
    
    /**
     * Public constructor.
     * @param propertiesTree Action properties.
     * @throws ConfigurationException Queue name not configured.
     * @throws JMSException Unable to configure JMS destination.
     * @throws NamingException Unable to configure JMS destination.
     */
    public JMSRouter(ConfigTree propertiesTree) throws ConfigurationException, NamingException, JMSException {
        super(propertiesTree);

        this.properties = propertiesTree; 
        logger.debug(properties);
        
        queueName = properties.getAttribute("jndiName");
        if(queueName == null) {
            throw new ConfigurationException("JMSRouter must specify a 'jndiName' property.");
        }
        
        boolean persistent = Boolean.parseBoolean( properties.getAttribute(PERSISTENT_ATTR, "true") );
        deliveryMode = persistent ? DeliveryMode.PERSISTENT : DeliveryMode.NON_PERSISTENT;
        
        String priorityStr = properties.getAttribute(PRIORITY_ATTR);
        if ( priorityStr != null )
	        priority = Integer.parseInt( priorityStr );
        
        final String ttlStr = properties.getAttribute(TIME_TO_LIVE_ATTR);
        if ( ttlStr != null )
	        timeToLive = Long.parseLong( ttlStr );
        
        createQueueSetup( queueName );
    }
    
    /**
	 * Will simply pass the message to the route method unmodified.
	 * @return <code>null</code> which will case the action pipeline processing to stop
	 */
    @Override
	public org.jboss.soa.esb.message.Message process( org.jboss.soa.esb.message.Message message ) throws ActionProcessingException
	{
    	route ( message );
    	
    	return null;
	}

    /* (non-Javadoc)
     * @see org.jboss.soa.esb.actions.routing.AbstractRouter#route(java.lang.Object)
     */
    public void route(Object message) throws ActionProcessingException {
    	
    	if(!(message instanceof org.jboss.soa.esb.message.Message)) {
            throw new ActionProcessingException("Cannot send Object [" + message.getClass().getName() + "] to destination [" + queueName + "]. Object must be an instance of org.jboss.soa.esb.message.Message) .");
        }
        
		final org.jboss.soa.esb.message.Message esbMessage = (org.jboss.soa.esb.message.Message)message;
        
        try {
        	Message jmsMessage = null;
        	
        	if ( unwrap ) {
				Object objectFromBody = getPayloadProxy().getPayload(esbMessage);
				jmsMessage = createJMSMessageWithObjectType( objectFromBody );
            } 
        	else  {
            	jmsMessage = createObjectMessage(Util.serialize(esbMessage));
        	}
        	
            setStringProperties(jmsMessage);
            setJMSProperties( esbMessage, jmsMessage );
            send( jmsMessage );
            
        } catch(Exception e) {
        	StringBuilder sb = new StringBuilder();
        	sb.append("Exception while sending message [").append(message).append("] to destination [");
            
            if (queueSetup != null)
            	sb.append(queueSetup.queueName).append("].");
            else
            	sb.append("null ].");
            
            String errorMessage = sb.toString();
            logger.error(errorMessage, e);
            throw new ActionProcessingException(errorMessage, e);
        }
    }
    
    protected Message createJMSMessageWithObjectType( Object objectFromBody ) throws JMSException
	{
		Message jmsMessage = null;
		if(objectFromBody instanceof String) {
        	jmsMessage = queueSetup.jmsSession.createTextMessage();

            if(logger.isDebugEnabled()) {
                logger.debug("Sending Text message: [" + objectFromBody + "] to destination [" + queueSetup.queueName + "].");
            }
            
            ((TextMessage)jmsMessage).setText((String)objectFromBody);
        } else if(objectFromBody instanceof byte[]) {
        	jmsMessage = queueSetup.jmsSession.createBytesMessage();
        
            if(logger.isDebugEnabled()) {
                logger.debug("Sending byte[] message: [" + objectFromBody + "] to destination [" + queueSetup.queueName + "].");
            }
            
            ((BytesMessage)jmsMessage).writeBytes((byte[])objectFromBody);
        } else {
        	jmsMessage = createObjectMessage(objectFromBody);
        }
		
		return jmsMessage;
	}

	protected void send( Message jmsMessage ) throws JMSException
	{
		queueSetup.jmsProducer.send(jmsMessage);
	}
	
	/**
	 * This method will set appropriate JMSProperties on the outgoing JMS Message instance.
	 * </p>
	 * Sublclasses can either override this method to add a different behaviour, or they can 
	 * set the strategy by calling {@link #setJmsPropertiesStrategy(JMSPropertiesSetter)}.
	 * </p> 
	 * See {@link org.jboss.soa.esb.notification.jms.JMSPropertiesSetter} for more info.
	 */
	protected void setJMSProperties(org.jboss.soa.esb.message.Message fromESBMessage, Message toJMSMessage ) throws JMSException { 	
		jmsPropertiesStrategy.setJMSProperties( fromESBMessage, toJMSMessage );
	}

	protected Message createObjectMessage(Object message) throws JMSException {
		Message jmsMessage;
		jmsMessage = queueSetup.jmsSession.createObjectMessage();
		
		if(logger.isDebugEnabled()) {
		    logger.debug("Sending Object message: [" + message + "] to destination [" + queueSetup.queueName + "].");
		}
		((ObjectMessage)jmsMessage).setObject((Serializable) message);
		return jmsMessage;
	}

    private void setStringProperties(Message msg) throws JMSException {
        String messagePropPrefix = "message-prop-";

        for(KeyValuePair property : properties.attributesAsList()) {
            String key = property.getKey();
            
            if(key.startsWith(messagePropPrefix) && key.length() > messagePropPrefix.length()) {
                msg.setStringProperty(key.substring(messagePropPrefix.length()), property.getValue());
            }
        }
    }
    
    /* (non-Javadoc)
     * @see org.jboss.soa.esb.actions.ActionProcessor#getOkNotification(java.lang.Object)
     */
    public Serializable getOkNotification(org.jboss.soa.esb.message.Message message) {
        return null;
    }

    /* (non-Javadoc)
     * @see org.jboss.soa.esb.actions.ActionProcessor#getErrorNotification(java.lang.Object)
     */
    public Serializable getErrorNotification(org.jboss.soa.esb.message.Message message) {
        return null;
    }

    @Override
    protected void finalize() throws Throwable {
        queueSetup.close();
        super.finalize();
    }
    
    protected void createQueueSetup( String queueName ) throws ConfigurationException
	{
		try {
			queueSetup = new JMSSendQueueSetup(queueName);
			queueSetup.setDeliveryMode( deliveryMode );
			queueSetup.setPriority( priority );
			queueSetup.setTimeToLive( timeToLive );
			if ( logger.isDebugEnabled() )
			{
				logger.debug( "JMSRouter DeliveryMode : " + deliveryMode);
				logger.debug( "JMSRouter Priority : " + priority);
				logger.debug( "JMSRouter TimeToLive : " + timeToLive);
			}
		} catch (Throwable t) {
			throw new ConfigurationException("Failed to configure JMS Queue for routing.", t);
		}
	}
    
    private static class JMSSendQueueSetup {
        JmsSession jmsSession;
        Queue jmsQueue;
        MessageProducer jmsProducer;
        String queueName;
        JmsConnectionPool pool;
        Properties environment;
        
        // TODO: Modify to support topic destinations too

        private JMSSendQueueSetup(String queueName) throws NamingException, JMSException, ConnectionException, NamingContextException  {
            environment = new Properties();
            environment.setProperty(Context.PROVIDER_URL, Configuration.getJndiServerURL());
            environment.setProperty(Context.INITIAL_CONTEXT_FACTORY, Configuration.getJndiServerContextFactory());
            environment.setProperty(Context.URL_PKG_PREFIXES, Configuration.getJndiServerPkgPrefix());
            Context oCtx = NamingContextPool.getNamingContext(environment);
            try {
                pool = JmsConnectionPoolContainer.getPool(environment, "ConnectionFactory");
                
                this.queueName = queueName;
                
                jmsSession = pool.getSession();
                boolean clean = true ;
                try {
                    try {
                    	jmsQueue = (Queue) oCtx.lookup(queueName);
                    } catch (NamingException ne) {
                        try {
                            oCtx = NamingContextPool.replaceNamingContext(oCtx, environment);
                            jmsQueue = (Queue) oCtx.lookup(queueName);
                        } catch (NamingException nex) {
                            //ActiveMQ
                            jmsQueue = jmsSession.createQueue(queueName);
                        }
                    }
                    jmsProducer = jmsSession.createProducer(jmsQueue);
                    clean = false ;
                } finally {
                    if (clean) {
                        pool.closeSession(jmsSession) ;
                    }
                }
            } finally {
                NamingContextPool.releaseNamingContext(oCtx) ;
            }
        }
        
        public void setDeliveryMode(final int deliveryMode ) throws JMSException
        {
        	if ( jmsProducer != null )
        		jmsProducer.setDeliveryMode( deliveryMode );
        }
        
        public void setPriority(final int priority ) throws JMSException
        {
        	if ( jmsProducer != null )
        		jmsProducer.setPriority( priority );
        }
        
        public void setTimeToLive(final long ttl ) throws JMSException
        {
        	if ( jmsProducer != null )
        		jmsProducer.setTimeToLive( ttl );
        }
        
        private void close() {
            try {
	            pool.closeSession(jmsSession);
                if (jmsProducer!=null) {
                    jmsProducer.close();                    
                }
            } catch (Exception e) {
                logger.error("Unable to close JMS Queue Setup.", e);
            } 
        }
    }

	protected void setJMSReplyTo( Message jmsMessage, org.jboss.soa.esb.message.Message esbMessage ) throws URISyntaxException, JMSException, NamingException, ConnectionException
	{
		EPR replyToEpr = esbMessage.getHeader().getCall().getReplyTo();
		if( !( replyToEpr instanceof JMSEpr) )
			return;
		
		JMSEpr jmsEpr = (JMSEpr) replyToEpr;
		String destinationType = jmsEpr.getDestinationType();
        Destination jmsDestination = null;
        
		if ( destinationType.equals( JMSEpr.QUEUE_TYPE ))
		{
            jmsDestination = queueSetup.jmsSession.createQueue( jmsEpr.getDestinationName() );
		}
		else
		{
			/* Topics are currently not supported.
            JmsConnectionPool pool = JmsConnectionPoolContainer.getPool(queueSetup.environment, "ConnectionFactory", JMSEpr.TOPIC_TYPE);
            jmsDestination = pool.getTopicSession().createTopic( jmsEpr.getDestinationName() );
            */
		}
		
		if ( jmsDestination != null )
			jmsMessage.setJMSReplyTo( jmsDestination );
		
	}

	/**
	 * The delivery mode in use.
	 * @return true if the delivery mode is DeliveryMode.PERSISTENT
	 */
	public boolean isDeliveryModePersistent()
	{
		return deliveryMode == DeliveryMode.PERSISTENT ;
	}
	
	/**
	 * The priority used when sending messages.
	 * 
	 * @return int	the priorty
	 */
	public int getPriority()
	{
		return priority;
	}
	
	/**
	 * The time-to-live used when sending messages.
	 * 
	 * @return int	the time-to-live for messages
	 */
	public long getTimeToLive()
	{
		return timeToLive;
	}

}
