/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

/**
 * 
 */
package org.jboss.internal.soa.esb.util;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.eprs.SFTPEpr;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.util.FileUtil;
import org.jboss.soa.esb.util.RemoteFileSystem;
import org.jboss.soa.esb.util.RemoteFileSystemException;

import com.jcraft.jsch.Channel;
import com.jcraft.jsch.ChannelSftp;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import com.jcraft.jsch.SftpATTRS;
import com.jcraft.jsch.SftpException;
import com.jcraft.jsch.UserInfo;
import com.jcraft.jsch.ChannelSftp.LsEntry;

/**
 * 
 * Implementation of sftp (Secure FTP over SSH) Based on JSch from JCraft
 * http://www.jcraft.com/
 * 
 * @author b_georges
 * 
 */

public class SecureFtpImpl implements RemoteFileSystem 
{

	private static final Logger _logger = Logger.getLogger(SecureFtpImpl.class);

	private static final String TMP_SUFFIX = ".rosettaPart";

	private static final String SECURE_CHANNEL = "sftp";

	// The objects implementing secure FTP over ssh
	private JSch m_oJSch = new JSch();

	private Session session = null;

	private Channel m_oChannel = null;

	private ChannelSftp m_oSftpChannel = null;

	// TODO Add support for certificate. Not for GA though.
	private URL m_oCertificate = null;

	@SuppressWarnings("unused")
	private boolean m_bConnected, m_bPassive;

	private int m_iPort;

	private SFTPEpr m_oEpr;

	private ConfigTree m_oParms;

	private String m_sFtpServer, m_sUser, m_sPasswd;

	private String m_sRemoteDir, m_sLocalDir;

	/*
	 * Constructor
	 * 
	 * @param p_oP Is a config treeThe used to initialize the object
	 * 
	 * @param connect If true create a new sftp session
	 * 
	 */
	public SecureFtpImpl(ConfigTree p_oP, boolean p_bConnect) throws ConfigurationException, RemoteFileSystemException
	{
		m_oParms = p_oP;
		initialize(p_bConnect);
	}

	/*
	 * Constructor
	 * 
	 * @param p_oP Is an EPR used to initialize the object
	 * 
	 * @param connect If true create a new sftp session
	 * 
	 */
	public SecureFtpImpl(SFTPEpr p_oP, boolean p_bConnect) throws ConfigurationException, RemoteFileSystemException 
	{

		m_oEpr = p_oP;

		URL url = null;
		try {
			url = m_oEpr.getURL();
		} catch (MalformedURLException e) {
			throw new RemoteFileSystemException(e);
		} catch (URISyntaxException e) {
			throw new RemoteFileSystemException(e);
		}

		m_sFtpServer = url.getHost();

		String[] sa = null;

		if (url.getUserInfo() != null)
			sa = url.getUserInfo().split(":");

		if (sa == null)
			sa = new String[] { "", "" };

		m_sUser = (sa.length < 1) ? "" : sa[0];
		m_sPasswd = (sa.length < 2) ? "" : sa[1];

		m_sRemoteDir = url.getFile();

		final String tmpdir = System.getProperty("java.io.tmpdir");
		if ((m_sRemoteDir == null) || (m_sRemoteDir.equals("")))
			m_sRemoteDir = ModulePropertyManager.getPropertyManager(
					ModulePropertyManager.TRANSPORTS_MODULE).getProperty(
							Environment.FTP_REMOTEDIR, tmpdir);

		m_iPort = url.getPort();
		if (m_iPort < 0)
			m_iPort = url.getDefaultPort();

		m_sLocalDir = ModulePropertyManager.getPropertyManager(
				ModulePropertyManager.TRANSPORTS_MODULE).getProperty(
						Environment.FTP_LOCALDIR, tmpdir);

		m_bPassive = false;

		try
		{
			m_bPassive = m_oEpr.getPassive();
		} catch (URISyntaxException e) 
		{
			_logger.warn(e);
		}

		try
		{
			m_oCertificate = m_oEpr.getCertificateURL();
		} 
		catch (MalformedURLException ex)
		{
			_logger.error(ex);
			
			throw new ConfigurationException(ex);
		}
		catch (URISyntaxException e) 
		{
			_logger.warn(e);
			
			throw new ConfigurationException(e);
		}

		// TODO there is still a bit of space for improvements here.
		configTreeFromEpr();

		initialize(p_bConnect);
	}

	/*
	 * Constructor
	 * 
	 * @param attribs The key/value pairs used to initialize the object
	 * 
	 * @param connect If true create a new sftp session using attribs
	 * 
	 */
	public SecureFtpImpl(List<KeyValuePair> attribs, boolean connect)
	throws ConfigurationException, RemoteFileSystemException 
	{
		m_oParms = new ConfigTree("fromProps");
		for (KeyValuePair oCurr : attribs)
			m_oParms.setAttribute(oCurr.getKey(), oCurr.getValue());
		initialize(connect);
	}

	private void initialize(boolean bConnect) throws ConfigurationException, RemoteFileSystemException
	{
		checkParms();
		
		if (bConnect) 
		{
			try
			{
				session = m_oJSch.getSession(m_sUser, m_sFtpServer, m_iPort);
	
				UserInfo ui = new SecureFtpUserInfo(m_sPasswd);
				session.setUserInfo(ui);
	
				session.connect();
	
				m_oChannel = session.openChannel(SECURE_CHANNEL);
	
				m_oChannel.connect();
	
				m_oSftpChannel = (ChannelSftp) m_oChannel;
	
				for (int i1 = 0; i1 < 10 && !session.isConnected(); i1++)
				{
					try
					{
						Thread.sleep(200);  // TODO magic number
					}
					catch (InterruptedException ex)
					{
						// try again?
					}
				}
				
				if (!session.isConnected())
					throw new RemoteFileSystemException("Can't connect to FTP server");
	
				m_bConnected = this.session.isConnected();
			}
			catch (JSchException ex)
			{
				_logger.error("Caught Secure FTP Exception.", ex);
				
				throw new RemoteFileSystemException(ex);
			}
		}
		// TODO set connection Mode [PASSIVE|ACTIVE]using m_bPassive ?

	}

	private void checkParms() throws ConfigurationException 
	{
		m_sFtpServer = m_oParms.getAttribute(PARMS_FTP_SERVER);
		if (null == m_sFtpServer)
			throw new ConfigurationException("No SFTP server specified");

		m_sUser = m_oParms.getAttribute(PARMS_USER);
		if (null == m_sUser)
			throw new ConfigurationException("No username specified for SFTP");

		m_sPasswd = m_oParms.getAttribute(PARMS_PASSWD);
		if (null == m_sPasswd)
			throw new ConfigurationException("No password specified for SFTP");

		m_sRemoteDir = m_oParms.getAttribute(PARMS_REMOTE_DIR);
		if (null == m_sRemoteDir)
			m_sRemoteDir = "";

		m_sLocalDir = m_oParms.getAttribute(PARMS_LOCAL_DIR);
		if (null == m_sLocalDir)
			m_sLocalDir = ".";

		String sAux = m_oParms.getAttribute(PARMS_PORT);
		
		try
		{
			m_iPort = (null == sAux) ? 22 : Integer.parseInt(sAux);
		}
		catch (Exception ex)
		{
			throw new ConfigurationException(ex);
		}

		m_bPassive = false;
		sAux = m_oParms.getAttribute(PARMS_PASSIVE);
		m_bPassive = (null != sAux) && Boolean.parseBoolean(sAux);

		final String certificate = m_oParms.getAttribute(PARMS_CERTIFICATE) ;
		
		if (certificate != null)
		{
			try
			{
				m_oCertificate = new URL(certificate);
			}
			catch (MalformedURLException ex)
			{
				throw new ConfigurationException(ex);
			}
		}
	}

	/*
	 * Deletes a file on the SFTP-Server
	 * 
	 * @param fileName The file's Name to be removed from the SFTP-Server
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#deleteRemoteFile(java.lang.String)
	 */
	public void deleteRemoteFile(String p_sFile) throws RemoteFileSystemException 
	{
		try
		{
			m_oSftpChannel.rm(getRemoteDir() + "/" + new File(p_sFile).getName());
		}
		catch (SftpException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	}

	/*
	 * Deletes a file on the SFTP-Server
	 * 
	 * @param fileName The file to be removed from the SFTP-Server
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#remoteDelete(java.io.File)
	 */
	public void remoteDelete(File p_oFile) throws RemoteFileSystemException 
	{
		try
		{
			m_oSftpChannel.rm(FtpUtils.fileToFtpString(p_oFile));
		}
		catch (SftpException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	}

	/*
	 * Returns a list of Filenames for the directory specified in p_Suffix
	 * 
	 * @param p_sSuffix The remote directory path from the SFTP-Server
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#getFileListFromRemoteDir(java.lang.String)
	 */
	public String[] getFileListFromRemoteDir(String p_sSuffix) throws RemoteFileSystemException 
	{
		try
		{
			String sSuffix = (null == p_sSuffix) ? "*" : "*" + p_sSuffix;
			List<String> lFileList = new ArrayList<String>();
			Vector vFileList = m_oSftpChannel.ls(sSuffix);
			
			if (vFileList != null) 
			{
				for (int i = 0; i < vFileList.size(); i++) 
				{
					Object obj = vFileList.elementAt(i);
					
					if (obj instanceof LsEntry) 
					{
						SftpATTRS oSftAttr = ((LsEntry) obj).getAttrs();
						if (!oSftAttr.isDir()) 
						{
							lFileList.add(((LsEntry) obj).getFilename());
						}
					}
				}
			}
			
			return (String[]) lFileList.toArray(new String[lFileList.size()]);
		}
		catch (SftpException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	}

	/*
	 * Set the new directory to p_SDir
	 * 
	 * @param p_sDir The remote directory path name we want to "cd" to.
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#setRemoteDir(java.lang.String)
	 */
	public void setRemoteDir(String p_sDir) throws RemoteFileSystemException 
	{
		try
		{
			m_oSftpChannel.cd(p_sDir);
		}
		catch (SftpException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	}

	/*
	 * Rename the Remote Directory name p_sFrom with p_sTo
	 * 
	 * @param p_sFrom The remote directory name we want to rename
	 * 
	 * @param p_sTo The new remote directory name
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#renameInRemoteDir(java.lang.String,
	 *      java.lang.String)
	 */
	public void renameInRemoteDir(String p_sFrom, String p_sTo)
	throws RemoteFileSystemException 
	{

		String sRmtFrom = new File(p_sFrom).getName();
		String sRmtTo = new File(p_sTo).getName();

		try 
		{
			m_oSftpChannel.rename(getRemoteDir() + "/" + sRmtFrom,
					getRemoteDir() + "/" + sRmtTo);
		} 
		catch (Exception e) 
		{
			String sMess = this.getClass().getSimpleName()
			+ " can't rename in remote directory <" + e.getMessage()
			+ ">";
			throw new RemoteFileSystemException(sMess);
		}
	}

	/*
	 * Rename the Remote File name p_sFrom with p_sTo
	 * 
	 * @param p_oFrom The remote file name we want to rename
	 * 
	 * @param p_oTo The new remote file name
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#remoteRename(java.io.File,
	 *      java.io.File)
	 */
	public void remoteRename(File p_oFrom, File p_oTo) throws RemoteFileSystemException 
	{
		try 
		{
			m_oSftpChannel.rename(FtpUtils.fileToFtpString(p_oFrom), FtpUtils
					.fileToFtpString(p_oTo));
		} 
		catch (Exception e) 
		{
			String sMess = this.getClass().getSimpleName()
			+ " can't rename in remote directory <" + e.getMessage()
			+ ">";
			throw new RemoteFileSystemException(sMess);
		}
	}

	/*
	 * Upload the local File p_ofile to p_sRemoteName
	 * 
	 * @param p_oFile The local file name we want to upload
	 * 
	 * @param p_sRemoteName The remote file name [can be the same as p_oFile of
	 * course]
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#uploadFile(java.io.File,
	 *      java.lang.String)
	 */
	public void uploadFile(File p_oFile, String p_sRemoteName) throws RemoteFileSystemException 
	{
		try
		{
			String sRemoteOK = getRemoteDir() + "/" + p_sRemoteName;
			String sRemoteTmp = sRemoteOK + TMP_SUFFIX;
			m_oSftpChannel.put(FtpUtils.fileToFtpString(p_oFile), sRemoteTmp);
			m_oSftpChannel.rename(sRemoteTmp, sRemoteOK);
		}
		catch (SftpException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	}

	/*
	 * Download the remote File p_sFile to p_sFile.
	 * 
	 * @param p_sFile The remote file name we want to download
	 * 
	 * @param p_sFinalName The local file name
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#downloadFile(java.lang.String,
	 *      java.lang.String)
	 */
	public void downloadFile(String p_sFile, String p_sFinalName) throws IOException, RemoteFileSystemException 
	{
		File oLocalDir = new File(m_sLocalDir);
		File oLclFile = File.createTempFile("Rosetta_", TMP_SUFFIX, oLocalDir);

		try 
		{
			oLclFile.delete();
		} 
		catch (Exception e) 
		{
			_logger.warn("Could not delete file: "+oLclFile, e);
		}
		// TODO check if we have to set the Transfer Type with JSch impl =>
		// m_oXferType
		
		try
		{
			m_oSftpChannel.get(p_sFile, FtpUtils.fileToFtpString(oLclFile));
	
			File oNew = new File(oLocalDir, p_sFinalName);
			if (oNew.exists())
				oNew.delete();
			FileUtil.renameTo(oLclFile, oNew);
		}
		catch (Exception ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	}

	/*
	 * Returns the current remote directory
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#getRemoteDir()
	 */
	public String getRemoteDir() 
	{
		return m_sRemoteDir;
	}

	/*
	 * Terminates the sftp session.
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#quit()
	 */
	public void quit() 
	{
		m_oSftpChannel.quit();
	}

	private void configTreeFromEpr() throws ConfigurationException 
	{
		m_oParms = new ConfigTree("fromEpr");
		try 
		{
			m_oParms.setAttribute(RemoteFileSystem.PARMS_FTP_SERVER,
					m_sFtpServer);
			m_oParms.setAttribute(RemoteFileSystem.PARMS_USER, m_sUser);
			m_oParms.setAttribute(RemoteFileSystem.PARMS_PASSWD, m_sPasswd);
			m_oParms.setAttribute(RemoteFileSystem.PARMS_REMOTE_DIR,
					m_sRemoteDir);
			m_oParms.setAttribute(RemoteFileSystem.PARMS_PORT, Integer
					.toString(m_iPort));
			m_oParms.setAttribute(RemoteFileSystem.PARMS_LOCAL_DIR, m_sLocalDir);
			m_oParms.setAttribute(RemoteFileSystem.PARMS_ASCII, Boolean
					.toString(false));
			m_oParms.setAttribute(RemoteFileSystem.PARMS_PASSIVE, Boolean
					.toString(m_bPassive));
			if (m_oCertificate != null)
			{
				m_oParms.setAttribute(RemoteFileSystem.PARMS_CERTIFICATE, m_oCertificate.toString());
			}
		} 
		catch (Exception e) 
		{
			throw new ConfigurationException(e);
		}
	}

}
