/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.util;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.List;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.eprs.FTPEpr;
import org.jboss.soa.esb.addressing.eprs.FileEpr;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.util.FileUtil;
import org.jboss.soa.esb.util.FtpClientUtil;
import org.jboss.soa.esb.util.RemoteFileSystem;
import org.jboss.soa.esb.util.RemoteFileSystemException;

import com.enterprisedt.net.ftp.FTPClient;
import com.enterprisedt.net.ftp.FTPConnectMode;
import com.enterprisedt.net.ftp.FTPException;
import com.enterprisedt.net.ftp.FTPTransferType;

/**
 * Simplified FTP transfers
 * <p>
 * Description: Implements a simple set of FTP functionality Parameters to
 * establish the FTP connection are provided at construction time and cannot
 * change during the lifetime of the object <br/>Hides low level details.
 * Current implementation is based on the "Entreprise Distributed Technology
 * edtFTPj" library but this can be changed with no impact to existing code,
 * just by changing this class without modifying the signature of it's public
 * methods
 * </p>
 */

public class EdtFtpImpl implements RemoteFileSystem
{

	private static final Logger _logger = Logger.getLogger(EdtFtpImpl.class);

	private static final String TMP_SUFFIX = ".rosettaPart";

	private boolean m_bPassive;

	private int m_iPort;
	private int _timeout = 0;

	private FTPClient m_oConn = new FTPClient();

	private FTPEpr m_oEpr;

	private ConfigTree m_oParms;

	private FTPTransferType m_oXferType;

	private String m_sFtpServer, m_sUser, m_sPasswd;

	private String m_sRemoteDir, m_sLocalDir;

	/**
	 * Checks validity and completeness of parameters, and keeps the info
	 * internally for subsequent FTP requests
	 * 
	 * @param p_oP
	 *            ConfigTree
	 * @throws ConfigurationException :
	 *             if parameters are invalid or incomplete
	 *             <li>Parameters: (XML attributes at the root level) </li>
	 *             <li> ftpServer = name or IP of FTP server </li>
	 *             <li> ftpUser = login ID for server </li>
	 *             <li> ftpPassword </li>
	 *             <li> localDirURI = absolute path in the local filesystem
	 *             </li>
	 *             <li> remoteDirURI = remote path is relative to ftp user home
	 *             in remote computer </li>
	 */

	public EdtFtpImpl (ConfigTree p_oP, boolean p_bConnect)
			throws ConfigurationException, RemoteFileSystemException
	{
		m_oParms = p_oP;
		initialize(p_bConnect);
	}

	public EdtFtpImpl (FTPEpr p_oP, boolean p_bConnect)
			throws ConfigurationException, RemoteFileSystemException
	{
		m_oEpr = p_oP;

		URL url = null;
		try
		{
			url = m_oEpr.getURL();
		}
		catch (MalformedURLException e)
		{
			throw new ConfigurationException(e);
		}
		catch (URISyntaxException e)
		{
			throw new ConfigurationException(e);
		}

		m_sFtpServer = url.getHost();

		String[] sa = null;

		if (url.getUserInfo() != null) sa = url.getUserInfo().split(":");

		if (sa == null) sa = new String[] { "", "" };

		m_sUser = (sa.length < 1) ? "" : sa[0];
		m_sPasswd = (sa.length < 2) ? "" : sa[1];

		m_sRemoteDir = url.getFile();

		final String tmpdir = System.getProperty("java.io.tmpdir");
		if ((m_sRemoteDir == null) || (m_sRemoteDir.equals("")))
			m_sRemoteDir = ModulePropertyManager.getPropertyManager(
					ModulePropertyManager.TRANSPORTS_MODULE).getProperty(
					Environment.FTP_REMOTEDIR, tmpdir);

		m_iPort = url.getPort();
		if (m_iPort < 0) m_iPort = url.getDefaultPort();

		m_sLocalDir = ModulePropertyManager.getPropertyManager(
				ModulePropertyManager.TRANSPORTS_MODULE).getProperty(
				Environment.FTP_LOCALDIR, tmpdir);

        File oLocalDir = new File(m_sLocalDir);
        if(!oLocalDir.exists()) {
            throw new ConfigurationException("Local FTP directory '" + oLocalDir.getAbsolutePath()
                    + "' doesn't exist.  Check your JBossESB config '"
                    + ModulePropertyManager.TRANSPORTS_MODULE + ":" + Environment.FTP_LOCALDIR + "'");
        }

        m_bPassive = false;

		try
		{
			m_bPassive = m_oEpr.getPassive();
		}
		catch (URISyntaxException e)
		{
			_logger.warn(e);
		}

		String timeout = ModulePropertyManager.getPropertyManager(ModulePropertyManager.TRANSPORTS_MODULE).getProperty(Environment.FTP_SOCKET_TIMEOUT, null);
		
		if (timeout != null)
		{
			try
			{
				_timeout = Integer.parseInt(timeout);
			}
			catch (NumberFormatException ex)
			{
				throw new ConfigurationException("Invalid timeout specified.", ex);
			}
		}
		else
			_timeout = 0;

		// TODO there is still a bit of space for improvements here.
		configTreeFromEpr();

		initialize(p_bConnect);
	}

	public EdtFtpImpl (List<KeyValuePair> p_oAttribs, boolean p_bConnect)
			throws ConfigurationException, RemoteFileSystemException
	{
		m_oParms = new ConfigTree("fromProps");
		for (KeyValuePair oCurr : p_oAttribs)
			m_oParms.setAttribute(oCurr.getKey(), oCurr.getValue());
		initialize(p_bConnect);
	}

	private void checkParms () throws ConfigurationException
	{
		String att = m_oParms.getAttribute(FileEpr.URL_TAG);
		URL url = null;
		
		try
		{
			if (att != null)
				url = new URL(att);
		}
		catch (MalformedURLException ex)
		{
			throw new ConfigurationException(ex);
		}
		
		m_sFtpServer = (null != url) ? url.getHost() : m_oParms
				.getAttribute(PARMS_FTP_SERVER);
		if (null == m_sFtpServer)
			throw new ConfigurationException("No FTP server specified");

		String[] sa = (null == url) ? null : url.getUserInfo().split(":");
		m_sUser = (null != sa) ? sa[0] : m_oParms.getAttribute(PARMS_USER);
		if (null == m_sUser)
			throw new ConfigurationException("No username specified for FTP");

		m_sPasswd = (null != sa) ? sa[1] : m_oParms.getAttribute(PARMS_PASSWD);
		if (null == m_sPasswd)
			throw new ConfigurationException("No password specified for FTP");

		m_sRemoteDir = (null != url) ? url.getFile() : m_oParms
				.getAttribute(PARMS_REMOTE_DIR);
		if (null == m_sRemoteDir) m_sRemoteDir = "";

		m_sLocalDir = m_oParms.getAttribute(PARMS_LOCAL_DIR);
		if (null == m_sLocalDir) m_sLocalDir = ".";

		String sAux = m_oParms.getAttribute(PARMS_PORT);
		m_iPort = (null != url) ? url.getPort() : (null == sAux) ? 21 : Integer
				.parseInt(sAux);
		
		try
		{
			if (m_iPort < 0)
				m_iPort = new URL("ftp://").getDefaultPort();
		}
		catch (MalformedURLException ex)
		{
			throw new ConfigurationException(ex);
		}

		// Dec-2006 (b_georges): it has been decided to set the Type to binary.
		m_oXferType = FTPTransferType.BINARY;

		m_bPassive = false;
		sAux = m_oParms.getAttribute(PARMS_PASSIVE);
		m_bPassive = (null != sAux) && Boolean.parseBoolean(sAux);

		return;
	}

	private void configTreeFromEpr () throws RemoteFileSystemException
	{
		m_oParms = new ConfigTree("fromEpr");
		try
		{
			m_oParms.setAttribute(RemoteFileSystem.PARMS_FTP_SERVER,
					m_sFtpServer);
			m_oParms.setAttribute(RemoteFileSystem.PARMS_USER, m_sUser);
			m_oParms.setAttribute(RemoteFileSystem.PARMS_PASSWD, m_sPasswd);
			m_oParms.setAttribute(RemoteFileSystem.PARMS_REMOTE_DIR,
					m_sRemoteDir);
			m_oParms.setAttribute(RemoteFileSystem.PARMS_PORT, Integer
					.toString(m_iPort));
			m_oParms
					.setAttribute(RemoteFileSystem.PARMS_LOCAL_DIR, m_sLocalDir);
			m_oParms.setAttribute(RemoteFileSystem.PARMS_ASCII, Boolean
					.toString(false));
			m_oParms.setAttribute(RemoteFileSystem.PARMS_PASSIVE, Boolean
					.toString(m_bPassive));
		}
		catch (Exception e)
		{
			throw new RemoteFileSystemException(e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#deleteRemoteFile(java.lang.String)
	 */
	public void deleteRemoteFile (String p_sFile) throws RemoteFileSystemException
	{
		try
		{
			m_oConn.delete(getRemoteDir() + "/" + new File(p_sFile).getName());
		}
		catch (Exception ex)
		{
			/*
			 * It seems as though we have a race condition whereby one thread
			 * tries to remove a file when another renames it!
			 */
			
			throw new RemoteFileSystemException(ex);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#downloadFile(java.lang.String,
	 *      java.lang.String)
	 */
	public void downloadFile (String p_sFile, String p_sFinalName)
			throws IOException, RemoteFileSystemException
	{
		File oLocalDir = new File(m_sLocalDir);

        File oLclFile = File.createTempFile("Rosetta_", TMP_SUFFIX, oLocalDir);

		try
		{
			oLclFile.delete();
		}
		catch (Exception e)
		{
		}

		try
		{
			m_oConn.get(FtpUtils.fileToFtpString(oLclFile), p_sFile);
		}
		catch (FTPException ex)
		{
			throw new RemoteFileSystemException(ex);
		}

                final File to = new File(p_sFinalName) ;
		final File oNew = (to.isAbsolute() ? to : new File(oLocalDir, p_sFinalName)) ;
		
		if (oNew.exists()) 
			oNew.delete();
		
		FileUtil.renameTo(oLclFile, oNew);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#getFileListFromRemoteDir(java.lang.String)
	 */
	public String[] getFileListFromRemoteDir (String p_sSuffix)
			throws RemoteFileSystemException, IOException
	{
		String sSuffix = (null == p_sSuffix) ? "*" : "*" + p_sSuffix;

		try
		{
			return m_oConn.dir(sSuffix);
		}
		catch (FTPException ex)
		{
			String msg = ex.getMessage();
			int rc = ex.getReplyCode();
			if (rc == 550) // means File Not Found - see JBESB-303
			{
				_logger
						.debug("No matching file or directory. Server returns: [" + rc + "] " + msg);
				return null;
			}
			else
			{
				// TODO Test with different FTP Servers
				String sMess = this.getClass().getSimpleName() + " can't list " + sSuffix + " due to: [" + rc + "] " + msg;
				throw new RemoteFileSystemException(sMess);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#getRemoteDir()
	 */
	public String getRemoteDir ()
	{
		return m_sRemoteDir;
	}

	private void initialize (boolean bConnect) throws ConfigurationException,
			RemoteFileSystemException
	{
		checkParms();
		if (bConnect)
		{
			try
			{
				m_oConn.setRemoteHost(m_sFtpServer);
			}
			catch (UnknownHostException ex)
			{
				_logger.error("Unknown host for FTP.", ex);

				throw new ConfigurationException(ex);
			}
			catch (IOException ex)
			{
				_logger.error("Caught IOException", ex);
				
				throw new RemoteFileSystemException(ex);
			}
			catch (FTPException ex)
			{
				_logger.error("Caught FTPException.", ex);

				throw new RemoteFileSystemException(ex);
			}

			try
			{
				m_oConn.setRemotePort(m_iPort);

				m_oConn.connect();
				for (int i1 = 0; i1 < 10 && !m_oConn.connected(); i1++)
				{
					try
					{
						Thread.sleep(200); // TODO arbitrary MAGIC number!
					}
					catch (InterruptedException ex)
					{
					}
				}

				// Configurable?
				if (!m_oConn.connected())
					throw new RemoteFileSystemException(
							"Can't connect to FTP server");
				m_oConn.user(m_sUser);
				m_oConn.password(m_sPasswd);
				
				/*
				 * It's ok to set the passive/active mode here as this is for
				 * the data connection. The previous call to connect() was for
				 * the command connection and is not affected by this setting.
				 * 
				 * http://www.slacksite.com/other/ftp.html
				 */
				
				m_oConn.setConnectMode((m_bPassive) ? FTPConnectMode.PASV : FTPConnectMode.ACTIVE);
				
				m_oConn.setType(m_oXferType);
				
				if (_timeout > 0)
				{
					try
					{ 
						m_oConn.setTimeout(_timeout); 
					}
					catch (IOException e)
					{ 
						throw new RemoteFileSystemException("Failed while setting timeout=" + _timeout, e); 
					}
				}
			}
			catch (IOException ex)
			{
				throw new RemoteFileSystemException(ex);
			}
			catch (FTPException ex)
			{
				_logger.error("Caught FTPException.", ex);

				throw new RemoteFileSystemException(ex);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#quit()
	 */
	public void quit ()
	{
		if (null != m_oConn)
		{
			try
			{
				m_oConn.quit();
			}
			catch (Exception e)
			{
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#remoteDelete(java.io.File)
	 */
	public void remoteDelete (File p_oFile) throws RemoteFileSystemException
	{
		try
		{
			m_oConn.delete(getRemoteDir() + "/" + p_oFile.getName());
		}
		catch (IOException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
		catch (FTPException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#remoteRename(java.io.File,
	 *      java.io.File)
	 */
	public void remoteRename (File p_oFrom, File p_oTo) throws RemoteFileSystemException
	{
		try
		{
			m_oConn.rename(FtpClientUtil.fileToFtpString(p_oFrom), FtpUtils
					.fileToFtpString(p_oTo));
		}
		catch (FTPException ex)
		{
			if (ex.getReplyCode() == 550) // EdtFtp error code meaning File
			// Not Found
			{
				_logger
						.debug("EdtFtpImpl tried to rename file that had moved.");

				throw new RemoteFileSystemException("File not found.");
			}
			else
			{
				_logger.error("Caught FTPException.", ex);

				throw new RemoteFileSystemException(ex);
			}
		}
		catch (Exception e)
		{
			e.printStackTrace();

			String sMess = this.getClass().getSimpleName() + " can't rename in remote directory <" + e
					.getMessage() + ">";
			throw new RemoteFileSystemException(sMess);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#renameInRemoteDir(java.lang.String,
	 *      java.lang.String)
	 */
	public void renameInRemoteDir (String p_sFrom, String p_sTo)
			throws RemoteFileSystemException
	{
		String sRmtFrom = new File(p_sFrom).getName();
		String sRmtTo = new File(p_sTo).getName();

		try
		{
			m_oConn.rename(getRemoteDir() + "/" + sRmtFrom,
					getRemoteDir() + "/" + sRmtTo);
		}
		catch (Exception e)
		{
			String sMess = this.getClass().getSimpleName() + " can't rename in remote directory <" + e
					.getMessage() + ">";
			throw new RemoteFileSystemException(sMess);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#setRemoteDir(java.lang.String)
	 */
	public void setRemoteDir (String p_sDir) throws RemoteFileSystemException
	{
		if (p_sDir == null)
			throw new IllegalArgumentException();
		
		try
		{
			m_oConn.chdir(p_sDir);
			m_sRemoteDir = p_sDir;
		}
		catch (IOException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
		catch (FTPException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.util.RemoteFileSystem#uploadFile(java.io.File,
	 *      java.lang.String)
	 */
	public void uploadFile (File p_oFile, String p_sRemoteName)
			throws RemoteFileSystemException
	{
		String sRemoteOK = getRemoteDir() + "/" + p_sRemoteName;
		String sRemoteTmp = sRemoteOK + TMP_SUFFIX;

		try
		{
			m_oConn.put(FtpUtils.fileToFtpString(p_oFile), sRemoteTmp);
			m_oConn.rename(sRemoteTmp, sRemoteOK);
		}
		catch (IOException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
		catch (FTPException ex)
		{
			throw new RemoteFileSystemException(ex);
		}
	}

}
