/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.services.registry;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jboss.soa.esb.common.Configuration;

import javax.xml.registry.Connection;
import javax.xml.registry.ConnectionFactory;
import javax.xml.registry.JAXRException;
import java.net.PasswordAuthentication;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;

/**
 * JAXR Connection Factory.
 * <p/>
 * Extracted from the {@link JAXRRegistryImpl}.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class JAXRConnectionFactory {

    private static Logger logger = Logger.getLogger(JAXRConnectionFactory.class);

    private Set<PasswordAuthentication> creds = new HashSet<PasswordAuthentication>();
    private Properties props = new Properties();

    public JAXRConnectionFactory() {
        props = new Properties();
        props.setProperty("javax.xml.registry.queryManagerURL", Configuration.getRegistryQueryManageURI());
        props.setProperty("javax.xml.registry.lifeCycleManagerURL", Configuration.getRegistryLifecycleManagerURI());
        props.setProperty("javax.xml.registry.factoryClass", Configuration.getRegistryFactoryClass());
        if (Configuration.getRegistrySemanticEquivalences()!=null) {
            props.setProperty("javax.xml.registry.semanticEquivalences", Configuration.getRegistrySemanticEquivalences());
        }
        if (Configuration.getRegistryPostalAddressScheme()!=null) {
            props.setProperty("javax.xml.registry.postalAddressScheme", Configuration.getRegistryPostalAddressScheme());
        }
        if (Configuration.getRegistrySecurityAuthenticationMethod()!=null) {
            props.setProperty("javax.xml.registry.security.authenticationMethod", Configuration.getRegistrySecurityAuthenticationMethod());
        }
        if (Configuration.getRegistryUDDIMaxRows()!=null) {
            props.setProperty("javax.xml.registry.uddi.maxRows", Configuration.getRegistryUDDIMaxRows());
        }
        if (Configuration.getRegistryScoutTransportClass()!=null) {
            props.setProperty("scout.proxy.transportClass", Configuration.getRegistryScoutTransportClass());
        }
        String user = Configuration.getRegistryUser();
        String password = Configuration.getRegistryPassword();
        PasswordAuthentication passwdAuth = new PasswordAuthentication(user, password.toCharArray());
        creds.add(passwdAuth);
    }


    /**
     * Creates a connecton to a JAXR capable registy.
     *
     * @return Connection to a Registry using JAXR.
     */
    protected Connection getConnection()
    {
        Connection connection = null;
        try
        {   // Create the connection, passing it the configuration properties
            ConnectionFactory factory = ConnectionFactory.newInstance();
            factory.setProperties(props);
            connection = factory.createConnection();
            connection.setCredentials(creds);
        } catch (JAXRException e) {
            logger.log(Level.ERROR, "Could not set up a connection to the Registry. " + e.getMessage(), e);
        }
        return connection;
    }

    /**
     * Closes the connection to the Registry
     */
    protected void closeConnection(Connection connection)
    {
        try {
            if (connection!=null && !connection.isClosed()) {
                connection.close();
            }
        } catch (JAXRException je) {
            logger.log(Level.ERROR, je.getMessage(), je);
        }
    }
}
