package org.jboss.internal.soa.esb.message.format;

import java.net.URI;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;

import org.jboss.internal.soa.esb.message.format.serialized.SerializedMessagePlugin;
import org.jboss.internal.soa.esb.message.format.xml.XMLMessagePlugin;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessagePlugin;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.util.ClassUtil;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/**
 * Messages may contain attachments that do not appear in the main payload body.
 * For example, binary document formats, zip files etc.
 * 
 * @author Mark Little
 */

public class MessageFactoryImpl extends MessageFactory
{

	public MessageFactoryImpl ()
	{
		reset();
	}

	public void reset ()
	{
		messageFormats.clear();
		
		/*
		 * Go through the properties loaded from the property file. Anything
		 * starting with MessagePlugin.MESSAGE_PLUGIN is assumed to be a plugin
		 * that we load and add to the list.
		 */
		
		Properties properties = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE).getProperties();
		
		if (properties != null)
		{
			Enumeration names = properties.propertyNames();

			while (names.hasMoreElements())
			{
				String attrName = (String) names.nextElement();
				
				if (attrName.startsWith(MessagePlugin.MESSAGE_PLUGIN))
				{
					try
					{
						String pluginName = properties.getProperty(attrName);
						Class c = ClassUtil.forName(pluginName, getClass());
						MessagePlugin thePlugin = (MessagePlugin) c.newInstance();

						messageFormats.put(thePlugin.getType(), thePlugin);
					}
					catch (ClassNotFoundException ex)
					{
						ex.printStackTrace();
					}
					catch (IllegalAccessException ex)
					{
						ex.printStackTrace();
					}
					catch (InstantiationException ex)
					{
						ex.printStackTrace();
					}
				}
			}
		}
	         
		/*
		 * Now add the default(s).
		 */
		
		messageFormats.put(MessageType.JBOSS_XML, new XMLMessagePlugin());
		messageFormats.put(MessageType.JAVA_SERIALIZED, new SerializedMessagePlugin());
	}
	
	public Message getMessage() // returns some default implementation.
	{
		return ((MessagePlugin) messageFormats.get(MessageType.DEFAULT_TYPE)).getMessage();
	}

	public Message getMessage(URI type) // returns a message of a specific type.
	{
		if (type == null)
			throw new IllegalArgumentException();

		MessagePlugin plugin = messageFormats.get(type);

		if (plugin != null)
			return plugin.getMessage();
		else
			return null;
	}

	public Message getMessage(Message msg, URI type) // convert a message
	// from one form to
	// another.
	{
		if ((msg == null) || (type == null))
			throw new IllegalArgumentException();

		return null;
	}

	public Object createBodyType (Message m, String bodyType)
	{
		if (m == null)
			throw new IllegalArgumentException();

		MessagePlugin plugin = messageFormats.get(m.getType());

		if (plugin != null)
			return plugin.createBodyType(m, bodyType);
		else
			throw new IllegalArgumentException();
	}
	
	private final Hashtable<URI, MessagePlugin> messageFormats = new Hashtable<URI, MessagePlugin>();

}
