/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.internal.soa.esb.dependencies;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.URL;
import java.util.Properties;

import javax.naming.InitialContext;

import org.apache.juddi.registry.rmi.JNDIRegistration;
import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.util.PropertiesHelper;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.system.ServiceMBeanSupport;
import org.jboss.system.server.ServerConfig;
import org.jboss.system.server.ServerConfigLocator;

public class JuddiRMIService extends ServiceMBeanSupport implements JuddiRMIServiceMBean
{
   private Logger logger = Logger.getLogger(this.getClass());
   public static final String PROPNAME_JAVA_NAMING_FACTORY_INITIAL = "java.naming.factory.initial";
   public static final String PROPNAME_JAVA_NAMING_PROVIDER_URL = "java.naming.provider.url";
   public static final String PROPNAME_JAVA_NAMING_FACTORY_URL_PKGS = "java.naming.factory.url.pkgs";
   
   private static final String DEFAULT_PROPERTIES_RESOURCE_FILE = "esb.juddi.xml";
   private static final String JUDDI_PROPERTIES_RESOURCE_FILE = "esb.juddi.properties";
   private static final String JUDDI_PROPERTY_FILE_COMMENTS = "Auto generated property file, do not edit" ;
   
   private String propertiesResourceFile ;

   public String getPropertiesResourceFile()
   {
      return propertiesResourceFile;
   }

   public void setPropertiesResourceFile(String propertiesResourceFile)
   {
      this.propertiesResourceFile = propertiesResourceFile;
   }
   
   protected void startService() throws Exception
	{
		logger.info("starting juddi RMI service");
		final String propertiesResourceFileVal ;
        if (propertiesResourceFile == null)
        {
            propertiesResourceFileVal = DEFAULT_PROPERTIES_RESOURCE_FILE ;
        }
        else
        {
            propertiesResourceFileVal = propertiesResourceFile ;
        }
        final File baseFile = new File(propertiesResourceFileVal) ;

        final InputStream xmlPropertyIS ;
        if (baseFile.isAbsolute())
        {
            xmlPropertyIS = new FileInputStream(baseFile);
        }
        else
        {
            final URL url = Thread.currentThread().getContextClassLoader().getResource(propertiesResourceFileVal);
            xmlPropertyIS = url.openStream() ;
        }
        final byte[] propertyFileContents ;
        try {
            final ByteArrayOutputStream baos = new ByteArrayOutputStream() ;
            XMLHelper.replaceSystemProperties(XMLHelper.getXMLStreamReader(xmlPropertyIS),
                XMLHelper.getXMLStreamWriter(baos)) ;
            propertyFileContents = baos.toByteArray() ;
        } finally {
            xmlPropertyIS.close() ;
        }
        
        final ServerConfig serverConfig = ServerConfigLocator.locate() ;
        final File dataDir = serverConfig.getServerDataDir() ;
        final File juddiPropertyFile = new File(dataDir, JUDDI_PROPERTIES_RESOURCE_FILE) ;
        
        final ByteArrayInputStream bais = new ByteArrayInputStream(propertyFileContents) ;
        final FileOutputStream juddiPropertyOS = new FileOutputStream(juddiPropertyFile) ;
        final Properties xmlProperties ;
        try {
            xmlProperties = PropertiesHelper.translateXMLToText(bais, juddiPropertyOS, JUDDI_PROPERTY_FILE_COMMENTS) ;
        } finally {
            juddiPropertyOS.close() ;
        }
        System.setProperty("juddi.propertiesFile", juddiPropertyFile.getAbsolutePath());
        System.setProperty("javax.xml.registry.ConnectionFactoryClass","org.apache.ws.scout.registry.ConnectionFactoryImpl");
        
        // Read properties from file and if they exist - pass them on to juddi as system properties
        String factoryInitial =  xmlProperties.getProperty(PROPNAME_JAVA_NAMING_FACTORY_INITIAL, null);
        String providerURL = xmlProperties.getProperty(PROPNAME_JAVA_NAMING_PROVIDER_URL, null); 
        String factoryURLPkgs = xmlProperties.getProperty(PROPNAME_JAVA_NAMING_FACTORY_URL_PKGS, null);
        if (factoryInitial != null) {
            System.setProperty(PROPNAME_JAVA_NAMING_FACTORY_INITIAL, factoryInitial);
        }
        if (providerURL != null) {
            System.setProperty(PROPNAME_JAVA_NAMING_PROVIDER_URL, providerURL);
        }
        if (factoryURLPkgs != null) {
            System.setProperty(PROPNAME_JAVA_NAMING_FACTORY_URL_PKGS, factoryURLPkgs);
        }
        JNDIRegistration.register();
	}


	protected void stopService() throws Exception
	{
		logger.info("Unbinding juddi services");
		final InitialContext ic = new InitialContext() ;
		ic.unbind(JNDIRegistration.INQUIRY_SERVICE) ;
		ic.unbind(JNDIRegistration.PUBLISH_SERVICE) ;
	}
}
