/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.dependencies;

import java.io.File;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DriverManager;

import org.jboss.system.ServiceMBeanSupport;
import org.jboss.system.server.ServerConfigLocator;

/**
 * Integration with H2.
 *
 * @author <a href="mailto:rickard.oberg@telkel.com">Rickard �berg</a>
 * @author <a href="mailto:Scott_Stark@displayscape.com">Scott Stark</a>.
 * @author <a href="mailto:pf@iprobot.com">Peter Fagerlund</a>
 * @author <a href="mailto:jason@planet57.com">Jason Dillon</a>
 * @author <a href="mailto:vesco.claudio@previnet.it">Claudio Vesco</a>
 * @author <a href="mailto:dimitris@jboss.org">Dimitris Andreadis</a>
 * @author <a href="mailto:kevin.conner@jboss.org">Kevin Conner</a>
 * @version $Revision: 18346 $
 */
public class H2Database extends ServiceMBeanSupport
   implements H2DatabaseMBean
{
   /** Default password: <code>empty string</code>. */
   private static final String DEFAULT_PASSWORD = "";
   
   /** Default user: <code>sa</code>. */
   private static final String DEFAULT_USER = "sa";
   
   /** JDBC Driver class: <code>org.h2.Driver</code>. */
   private static final String JDBC_DRIVER_CLASS = "org.h2.Driver";
   
   /** JDBC URL common prefix: <code>jdbc:h2:</code>. */
   private static final String JDBC_URL_PREFIX = "jdbc:h2:";
   
   /** JDBC in memory URL prefix: <code>jdbc:h2:mem:</code>. */
   private static final String JDBC_MEM_URL_PREFIX = JDBC_URL_PREFIX + "mem:";
   
   /** Default data subdir: <code>h2</code>. */
   private static final String H2_DATA_DIR = "h2";
   
   /** Default database name: <code>default</code>. */
   private static final String DEFAULT_DATABASE_NAME = "default";

   // Private Data --------------------------------------------------
   
   /** Full path to db/h2. */
   private File dbPath;

   /** Database name. */
   private String name = DEFAULT_DATABASE_NAME;
   
   /** In memory mode. */
   private boolean inMemoryMode ;
   
   /** Database user. */
   private String user = DEFAULT_USER;
   
   /** Database password. */
   private String password = DEFAULT_PASSWORD;
   
   /** Hold a connection for in memory h2. */
   private Connection connection;


   // Attributes ----------------------------------------------------
   
   /**
    * Set the database name.
    * 
    * @jmx.managed-attribute
    */
   public void setDatabase(String name)
   {
      if (name == null)
      {
         name = DEFAULT_DATABASE_NAME;
      }
      this.name = name;
   }

   /**
    * Get the database name.
    * 
    * @jmx.managed-attribute
    */
   public String getDatabase()
   {
      return name;
   }

   /**
    * Get the full database path.
    * 
    * @jmx.managed-attribute
    */
   public String getDatabasePath()
   {
      if (dbPath != null)
      {
         return dbPath.toString();
      }
      else
      {
         return null;
      }
   }

   /**
    * @return the <code>inMemoryMode</code> flag.
    * 
    * @jmx.managed-attribute 
    */
   public boolean isInMemoryMode()
   {
      return inMemoryMode;
   }

   /**
    * If <b>true</b> the h2 is in memory mode otherwise h2 is in server or remote mode.
    * 
    * @param b in memory mode.
    * 
    * @jmx.managed-attribute
    */
   public void setInMemoryMode(boolean b)
   {
      inMemoryMode = b;
   }

   /**
    * @return the password
    * 
    * @jmx.managed-attribute 
    */
   public String getPassword()
   {
      return password;
   }

   /**
    * @return the user
    * 
    * @jmx.managed-attribute 
    */
   public String getUser()
   {
      return user;
   }

   /**
    * @param password
    * 
    * @jmx.managed-attribute 
    */
   public void setPassword(String password)
   {
      if (password == null)
      {
         password = DEFAULT_PASSWORD;
      }
      this.password = password;
   }

   /**
    * @param user
    * 
    * @jmx.managed-attribute 
    */
   public void setUser(String user)
   {
      if (user == null)
      {
         user = DEFAULT_USER;
      }
      this.user = user;
   }

   // Lifecycle -----------------------------------------------------
   
   /**
    * Start the database
    */
   protected void startService() throws Exception
   {
      if (inMemoryMode)
      {
          startInMemoryDatabase();
      }
      else
      {
          startStandaloneDatabase();
      }
   }

   /**
    * We now close the connection clean by calling the
    * serverSocket throught jdbc. The MBeanServer calls this 
    * method at closing time.
    */
   protected void stopService() throws Exception
   {
       if (inMemoryMode)
       {
           stopInMemoryDatabase();
       }
       else
       {
           stopStandaloneDatabase();
       }
   }
   
   // Private -------------------------------------------------------
   
   /**
    * Start the standalone (in process) database.
    */
   private void startStandaloneDatabase() throws Exception
   {
       final File h2Dir = checkDataDir() ;
       
       dbPath = new File(h2Dir, name);

       final String dbURL = JDBC_URL_PREFIX + dbPath.toURI().toString();

       // Check wee have connectivity
       final Connection connection = getConnection(dbURL);
       connection.close() ;
   }

   /**
    * Start the only in memory database.
    */
   private void startInMemoryDatabase() throws Exception
   {
       final String dbURL = JDBC_MEM_URL_PREFIX + name;

       // hold a connection so h2 does not close the database
       connection = getConnection(dbURL);
   }

   /**
    * Stop the standalone (in process) database.
    */
   private void stopStandaloneDatabase() throws Exception
   {
       // Nothing to do here
       log.info("Database standalone closed clean");
   }

   /**
    * Stop the in memory database.
    */
   private void stopInMemoryDatabase() throws Exception
   {
       try
       {
           connection.close() ;
       }
       finally
       {
           connection = null;
       }
       log.info("Database in memory closed clean");
   }
   
   /**
    * Get the connection.
    * 
    * @param dbURL jdbc url.
    * @return the connection, allocate one if needed.
    * @throws Exception
    */
   private synchronized Connection getConnection(String dbURL) throws Exception
   {
      if (connection == null)
      {
         ClassLoader cl = Thread.currentThread().getContextClassLoader();
         Class.forName(JDBC_DRIVER_CLASS, true, cl).newInstance();
         connection = DriverManager.getConnection(dbURL, user, password);
      }
      return connection;
   }
   
   /**
    * Check the existence of the h2 data directory.
    * @return The h2 data directory.
    * @throws IOException For errors checking/creating the h2 data directory.
    */
   private File checkDataDir()
       throws IOException
   {
       // Get the server data directory
       final File dataDir = ServerConfigLocator.locate().getServerDataDir();

       // Get DB directory
       final File h2Dir = new File(dataDir, H2_DATA_DIR);

       if (!h2Dir.exists())
       {
           h2Dir.mkdirs();
       }
       else if (!h2Dir.isDirectory())
       {
          throw new IOException("Failed to create directory: " + h2Dir);
       }
       return h2Dir ;
   }

}
