/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.couriers;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.helpers.JDBCEprDBResourceFactory;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.eprs.JDBCEpr;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;
import org.jboss.soa.esb.couriers.*;
import org.jboss.soa.esb.listeners.message.errors.Factory;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.util.Type;
import org.jboss.soa.esb.util.Util;

import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

public class SqlTableCourier implements PickUpOnlyCourier, DeliverOnlyCourier
{
    protected long _pollLatency = 200;

    protected long _sleepForRetries = 3000; // milliseconds

    protected boolean deleteOnSuccess, deleteOnError;
	protected boolean _isReceiver;

    private JDBCEprDBResourceFactory jdbcFactory;

	protected static Logger _logger = Logger.getLogger(SqlTableCourier.class);

	/**
	 * package protected constructor - Objects of Courier should only be
	 * instantiated by the Factory
	 * 
	 * @param epr
	 */
	SqlTableCourier(JDBCEpr epr) throws CourierException
	{
		this(epr, false);
    }

	/**
	 * package protected constructor - Objects of Courier should only be
	 * instantiated by the Factory
	 * 
	 * @param epr
	 */
	SqlTableCourier(JDBCEpr epr, boolean isReceiver) throws CourierException
	{
		_isReceiver = isReceiver;
		_sleepForRetries = 3000;  // TODO magic number - configurable?
		try
		{
			deleteOnSuccess = Boolean.TRUE.equals(Boolean.valueOf(epr
					.getPostDelete()));
			deleteOnError = Boolean.TRUE.equals(Boolean.valueOf(epr
					.getErrorDelete()));
		}
		catch (URISyntaxException e)
		{
			throw new CourierException(e);
		}

        jdbcFactory = new JDBCEprDBResourceFactory(epr);
	}

	public void cleanup() {
	}

	/**
	 * package the ESB message in a java.io.Serializable, and write it.
	 * Delivery occurs within its own transaction if there is no
	 * global transaction active.
	 * 
	 * @param message
	 *            Message - the message to deliverAsync
	 * @return boolean - the result of the delivery
	 * @throws CourierException -
	 *             if problems were encountered
	 */
	
	public boolean deliver(Message message) throws CourierException
	{
		if (_isReceiver)
			throw new CourierException("This is a read-only Courier");

		if (null == message)
			return false;

		String msgId;
		Call call = message.getHeader().getCall();
		if (null==call)
			message.getHeader().setCall(call=new Call());
		try
		{
			if (null==call.getMessageID())
				call.setMessageID(new URI(UUID.randomUUID().toString()));
			msgId = call.getMessageID().toString();
		}
		catch (URISyntaxException e)
		{
			throw new CourierException("Problems with message header ",e);
		}

        boolean transactional = isTransactional();

        Serializable serilaizedMessage;
        try {
            serilaizedMessage = Util.serialize(message);
        } catch (Exception e) {
            throw new CourierTransportException("Unable to serialize ESB Message.", e);
        }

        Connection connection = jdbcFactory.createConnection(transactional);
        try
        {
            PreparedStatement insertStatement = jdbcFactory.createInsertStatement(connection);
            try {
                insertStatement.setString(1, msgId);
                insertStatement.setObject(2, serilaizedMessage);
                insertStatement.setString(3, State.Pending.getColumnValue());
                insertStatement.setLong(4, System.currentTimeMillis());

                insertStatement.executeUpdate();
            } finally {
                insertStatement.close();
            }

            if (!transactional) {
                connection.commit();
            }

            return true;
        }
        catch (SQLException e)
        {
            try
            {
                if (!transactional) {
                    connection.rollback();
                }
            }
            catch (Exception roll)
            {
                _logger.debug(roll);
            }

            _logger.debug("SQL exception during deliver", e);
            throw new CourierTransportException(e);
        } finally {
            try {
                if (!transactional) {
                    connection.close();
                }
            } catch (SQLException e) {
                _logger.error("Exception while closing DataSource connection.", e);
            }
        }
	}

    public Message pickup(long millis) throws CourierException, CourierTimeoutException
	{
		Message result = null;
		long limit = System.currentTimeMillis()
				+ ((millis < 100) ? 100 : millis);

		do
		{
            boolean transactional = isTransactional();
            Connection connection = jdbcFactory.createConnection(transactional);
            try {
                PreparedStatement listStatement = jdbcFactory.createListStatement(connection);
                try {
                    ResultSet resultSet = listStatement.executeQuery();
                    try {
                        while (resultSet.next()) {
                            String messageId = resultSet.getString(1);

                            result = tryToPickup(messageId, connection);

                            // We've successfully picked up a message, so we can commit on a
                            // non-transacted connection...
                            if (!transactional) {
                                connection.commit();
                            }

                            if (result != null) {
                                /*
                                 * If this is fault message, then throw an exception with the contents. With the
                                 * exception of user-defined exceptions, faults will have nothing in the body, properties etc.
                                 */
                                if (Type.isFaultMessage(result)) {
                                    Factory.createExceptionFromFault(result);
                                } else {
                                    return result;
                                }
                            }
                        }
                    } finally {
                        try {
                            resultSet.close();
                        } catch (Exception e) {
                            _logger.warn("SQL Exception closing ResultSet", e);
                        }
                    }
                } finally {
                    try {
                        listStatement.close();
                    } catch (Exception e) {
                        _logger.warn("SQL Exception closing PreparedStatement", e);
                    }
                }
            } catch (FaultMessageException e) {
                // The picked up message was a fault, generating this exception
                // in Factory.createExceptionFromFault.  Just rethrow...
                throw e;
            } catch (Exception e) {
                _logger.warn("Exception during pickup", e);
                if (!transactional) {
                    try {
                        connection.rollback();
                    } catch (SQLException e1) {
                        _logger.warn("SQL Exception during rollback", e);
                    }
                }
                throw new CourierTransportException(e);
            } finally {
                try {
                    connection.close();
                } catch (SQLException e) {
                    _logger.warn("Error closing DataSource Connection.", e);
                }
            }

            try {
                long lSleep = limit - System.currentTimeMillis();
                if (_pollLatency < lSleep)
                    lSleep = _pollLatency;
                if (lSleep > 0)
                    Thread.sleep(lSleep);
            }
            catch (InterruptedException e) {
                return null;
            }
        } while (System.currentTimeMillis() <= limit);

        return null;
    }

    private Message tryToPickup(String messageId, Connection connection) throws CourierException, SQLException
	{
        PreparedStatement selectUpdateStatement = jdbcFactory.createSelect4UpdateStatement(connection);

        try {
            selectUpdateStatement.setString(1, messageId);
            selectUpdateStatement.setString(2, State.Pending.getColumnValue());

            ResultSet resultSet = selectUpdateStatement.executeQuery();
            try
            {
                if (resultSet.next())
                {
                    Message result = null;

                    try
                    {
                        Serializable blob = (Serializable) resultSet.getObject(1);
                        result = Util.deserialize(blob);
                    }
                    catch (Exception e)
                    {
                        // If there's an error deserializing the message blob, we either
                        // delete the message (deleteOnError), or change it's state
                        // to "State.Error" i.e. no exceptions/rollbacks...
                        result = null;
                    } finally {
                        if (result == null && deleteOnError) {
                            deleteMsg(messageId, connection);
                        } else if (result != null && deleteOnSuccess) {
                            deleteMsg(messageId, connection);
                        } else if(result == null) {
                            changeStatus(messageId, State.Error, connection);
                        } else {
                            changeStatus(messageId, State.Done, connection);
                        }
                    }

                    return result;
                }
            }
            finally
            {
                try
                {
                    resultSet.close();
                } catch (final Exception ex) {
                    _logger.warn("Could not close ResultSet.", ex);
                }
            }
        } finally {
            selectUpdateStatement.close();
        }

        return null;
	}

    private void deleteMsg(String messageId, Connection connection) throws SQLException
	{
        PreparedStatement statement = jdbcFactory.createDeleteStatement(connection);

        try {
            statement.setString(1, messageId);
            statement.executeUpdate();
        }   finally {
            statement.close();
        }
    }

    private void changeStatus(String messageId, State to, Connection connection) throws SQLException
	{
        PreparedStatement statement = jdbcFactory.createUpdateStatusStatement(connection);

        try {
            statement.setString(1, to.getColumnValue());
            statement.setString(2, messageId);
            statement.executeUpdate();
        } finally {
            statement.close();
        }
    }

    public static enum State
	{
		Pending, WorkInProgress, Done, Error;

        public String getColumnValue()
		{
			return toString().substring(0, 1);
		}

    }

    public void setPollLatency(Long millis)
	{
		if (millis <= 200)
			_logger.warn("Poll latency must be >= 200 milliseconds - Keeping old value of "+_pollLatency);
		else
			_pollLatency = millis;
	}

    private boolean isTransactional() throws CourierException {
        boolean transactional;
        try
        {
            TransactionStrategy txStrategy = TransactionStrategy.getTransactionStrategy(true);
            Object txHandle = ((txStrategy == null) ? null : txStrategy.getTransaction());
            boolean isActive = ((txStrategy == null) ? false : txStrategy.isActive());

            transactional = (txHandle != null);

            /*
            * Make sure the current transaction is still active! If we
            * have previously slept, then the timeout may be longer than that
            * associated with the transaction.
            */

            if (transactional && !isActive)
            {
                throw new CourierException("Associated transaction is no longer active!");
            }
        }
        catch (TransactionStrategyException ex)
        {
            throw new CourierException(ex);
        }
        return transactional;
    }
}