/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.couriers;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.hibernate.Query;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.addressing.eprs.HibernateEpr;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierTimeoutException;
import org.jboss.soa.esb.message.Message;

/**
 * Courier which delivers message.
 * 
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 * @since Version 4.2
 * 
 */
public class HibernateCourier implements PickUpOnlyCourier, DeliverOnlyCourier {

	protected HibernateEpr m_epr = null;
	protected static Logger m_Logger = Logger.getLogger(HibernateCourier.class);

	protected Query m_prepDelete = null;
	protected Query m_prepInsert = null;
	protected Query m_prepUpdate = null;
	protected Query m_prepSelUpd = null;
	protected Query m_prepGetList = null;
	
	private boolean m_isReceiver = false;
	protected long m_pollLatency = 200;

	
	private HibernateCourier() {
	}
	
	/**
	 * package protected constructor - Objects of Courier should only be
	 * instantiated by the Factory
	 * 
	 * @param epr
	 */
	HibernateCourier(HibernateEpr epr) throws CourierException {
		this(epr, false);
	}
	
	public HibernateCourier(HibernateEpr epr, boolean pickUpOnly) {
		m_isReceiver = pickUpOnly;
		m_epr = epr;		
	}

	public void cleanup() {
	}

	/**
	 * package the ESB message in a java.io.Serializable, and write it
	 * 
	 * @param message
	 *            Message - the message to deliverAsync
	 * @return boolean - the result of the delivery
	 * @throws CourierException -
	 *             if problems were encountered
	 */
	public boolean deliver(Message message) throws CourierException, MalformedEPRException {
		if (m_isReceiver)
			throw new CourierException("This is a read-only Courier");

		if (null == message)
			return false;

		/*
		 * This method doesn't really do anything about delivering
		 * the message. And pickup doesn't return a message either.
		 */
		
		Call call = message.getHeader().getCall();
		if (null==call)
			message.getHeader().setCall(call=new Call());
		try {
			if (null == call.getMessageID())
				call.setMessageID(new URI(UUID.randomUUID().toString()));
		} catch (URISyntaxException e) {
			throw new MalformedEPRException("Problems with message header ",e);
		}

		return false;
	}
	
	
	/* (non-Javadoc)
	 * @see org.jboss.internal.soa.esb.couriers.PickUpOnlyCourier#pickup(long)
	 */
	
	public Message pickup(long millis) throws CourierException,
			CourierTimeoutException {
		return null;
	}
	

	
	protected enum State {
		Pending, WorkInProgress, Done, Error;
		String getColumnValue() {
			return toString().substring(0, 1);
		}
	}
}
