/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/*
 * NAME
 *	rename10
 *
 * DESCRIPTION
 *	This test will verify that rename(2) syscall fails with ENAMETOOLONG
 *      and ENOENT
 *
 * ALGORITHM
 *	Setup:
 *		Setup signal handling.
 *		Create temporary directory.
 *		Pause for SIGUSR1 if option specified.
 *              create the "old" file
 *
 *	Test:
 *		Loop if the proper options are given.
 *              1.  rename the "old" to the "new" file
 *                  verify rename() failed with error ENAMETOOLONG
 *
 *              2.  "new" path contains a directory that does not exist
 *                  rename the "old" to the "new"
 *                  verify rename() failed with error ENOENT
 *	Cleanup:
 *		Print errno log and/or timing stats if options given
 *		Delete the temporary directory created.*
 *
 * USAGE
 *	rename10 [-c n] [-e] [-i n] [-I x] [-P x] [-t]
 *	where,  -c n : Run n copies concurrently.
 *		-e   : Turn on errno logging.
 *		-i n : Execute test n times.
 *		-I x : Execute test for x seconds.
 *		-P x : Pause for x seconds between iterations.
 *		-t   : Turn on syscall timing.
 *
 * HISTORY
 *	07/2001 Ported by Wayne Boyer
 *
 * RESTRICTIONS
 *	None.
 */
#include <sys/types.h>
#include <sys/fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "test.h"
#include "usctest.h"

void setup();
void cleanup();
extern void do_file_setup(char *);

char *TCID="rename10";		/* Test program identifier.    */
int TST_TOTAL=2;		/* Total number of test cases. */
extern int Tst_count;		/* Test Case counter for tst_* routines */

char badmname[] = "abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstmnopqrstuvwxyzabcdefghijklmnopqrstmnopqrstuvwxyzabcdefghijklmnopqrstmnopqrstuvwxyzabcdefghijklmnopqrstmnopqrstuvwxyzabcdefghijklmnopqrstmnopqrstuvwxyzabcdefghijklmnopqrstmnopqrstuvwxyz";

int exp_enos[]={ENAMETOOLONG, ENOENT, 0};     /* List must end with 0 */

int fd;
char fname[255], mname[255];
char mdir[255];

struct test_case_t {
	char *fd1;
	char *fd2;
	int error;
} TC[] = {
	/* badmname is too long for a file name - ENAMETOOLONG */
	{fname, badmname, ENAMETOOLONG},

	/* mname contains a directory component which does not exist - ENOENT */
	{fname, mname, ENOENT}
};

int
main(int ac, char **av)
{
	int lc;             /* loop counter */
	const char *msg;    /* message returned from parse_opts */
	int i;

	/*
	 * parse standard options
	 */
	if ((msg=parse_opts(ac, av, (option_t *)NULL, NULL)) != (char *)NULL) {
		tst_brkm(TBROK, tst_exit, "OPTION PARSING ERROR - %s", msg);
	}

	/*
	 * perform global setup for test
	 */
	setup();
	
	/* set the expected errnos... */
	TEST_EXP_ENOS(exp_enos);
	
	/*
	 * check looping state if -i option given
	 */
	for (lc=0; TEST_LOOPING(lc); lc++) {
	  
		/* reset Tst_count in case we are looping. */
		Tst_count=0;

		/* loop through the test cases */
		for (i=0; i < TST_TOTAL; i++) {

			TEST(rename(TC[i].fd1, TC[i].fd2));

			if (TEST_RETURN != -1) {
				tst_resm(TFAIL, "call succeeded unexpectedly");
				continue;
			}

			TEST_ERROR_LOG(TEST_ERRNO);

			if (TEST_ERRNO == TC[i].error) {
				tst_resm(TPASS, "expected failure - "
					 "errno = %d : %s", TEST_ERRNO,
					 strerror(TEST_ERRNO));
			} else {
				tst_resm(TFAIL, "unexpected error - %d : %s - "
					 "expected %d", TEST_ERRNO,
					 strerror(TEST_ERRNO), TC[i].error);
			}
		}
	}   /* End for TEST_LOOPING */
	
	/*
	 * cleanup and exit
	 */
	cleanup();

	/*NOTREACHED*/	
}

/*
 * setup() - performs all ONE TIME setup for this test.
 */
void 
setup()
{
	/* capture signals */
	tst_sig(NOFORK, DEF_HANDLER, cleanup);

	/* Pause if that option was specified */
	TEST_PAUSE; 

	/* Create a temporary directory and make it current. */
	tst_tmpdir();
	
	sprintf(fname,"./tfile_%d",getpid());
	sprintf(mdir,"./rndir_%d",getpid());
	sprintf(mname,"%s/rnfile_%d",mdir,getpid());

	do_file_setup(fname);
}

/*
 * cleanup() - performs all ONE TIME cleanup for this test at
 *             completion or premature exit.
 */
void 
cleanup()
{
	/*
	 * print timing stats if that option was specified.
	 * print errno log if that option was specified.
	 */
	TEST_CLEANUP;

	/*
	 * Remove the temporary directory.
	 */
	tst_rmdir();
	
	/*
	 * Exit with return code appropriate for results.
	 */
	tst_exit();
}
