/*
   (c) Copyright 2002-2003  Denis Oliver Kropp <dok@directfb.org>
   All rights reserved.

   XDirectFB is mainly based on XDarwin and
   also contains some KDrive, XFree and XWin code.
*/
/**************************************************************
 *
 * Support for using the DirectFB cursor
 *
 **************************************************************/
/* $XFree86: xc/programs/Xserver/hw/directfb/bundle/directfbCursor.c,v 1.15 2001/12/22 05:28:35 torrey Exp $ */

#include "directfbScreen.h"
#include "directfbCursor.h"
#include "directfbX.h"
#include "rootlessDirectFB.h"

#include "mi.h"
#include "scrnintstr.h"
#include "cursorstr.h"
#include "mipointrst.h"
#include "servermd.h"
#include "globals.h"

typedef struct {
     QueryBestSizeProcPtr    QueryBestSize;
     miPointerSpriteFuncPtr  spriteFuncs;
} DirectFBCursorScreenRec, *DirectFBCursorScreenPtr;

static int directfbCursorScreenIndex = -1;
static unsigned long directfbCursorGeneration = 0;

#define CURSOR_PRIV(pScreen) \
    ((DirectFBCursorScreenPtr)pScreen->devPrivates[directfbCursorScreenIndex].ptr)

#define SHADOW_XY(x,y) (((x) >= 0 && (y) >= 0 && (x) < full_width && (y) < full_height) ? shadow_data[(x) + (y) * full_width] : 0)

/*
 * MakeCursorSurface
 */
static IDirectFBSurface *
MakeCursorSurface (CursorPtr pCursor)
{
     DFBResult              ret;
     DFBSurfaceDescription  dsc;
     IDirectFBSurface      *surface;
     __u32                 *data, *dst;
     int                    pitch;
     int                    x, y;
     int                    width, height;
     int                    full_width, full_height;
     int                    cursor_pitch;
     __u32                  colors[2]; /* foreground/background in ARGB */
     __u8                  *shadow_data;
     __u8                  *shadow_dst;

     /* remember locally */
     width  = pCursor->bits->width;
     height = pCursor->bits->height;

     full_width  = width;
     full_height = height;

#ifdef ARGB_CURSOR
     if (!pCursor->bits->argb)
#endif
     {
          /* add shadow */
          full_width  += 4;
          full_height += 4;
     }

     /* fill surface description for cursor shape */
     dsc.flags       = DSDESC_WIDTH | DSDESC_HEIGHT | DSDESC_PIXELFORMAT;
     dsc.width       = full_width;
     dsc.height      = full_height;
     dsc.pixelformat = DSPF_ARGB;

     /* create the surface containing the cursor shape */
     ret = dfb->CreateSurface (dfb, &dsc, &surface);
     if (ret) {
          DirectFBError ("MakeCursorSurface: CreateSurface", ret);
          return NULL;
     }

     /* lock surface for writing */
     ret = surface->Lock (surface, DSLF_WRITE, (void**)&data, &pitch);
     if (ret) {
          DirectFBError ("MakeCursorSurface: Surface::Lock", ret);
          surface->Release (surface);
          return NULL;
     }

#ifdef ARGB_CURSOR
     if (pCursor->bits->argb) {
          CARD32 *src = pCursor->bits->argb;
          dst = data;

          /* render cursor image */
          for (y=0; y<height; y++) {
               for (x=0; x<width; x++) {
                    dst[x] = src[x];
               }

               /* next line */
               src += width;
               dst += pitch / 4;
          }

          /* unlock before returning */
          surface->Unlock (surface);

          return surface;
     }
#endif

     /* allocate shadow data */
     shadow_data = alloca (full_width * full_height);
     memset (shadow_data, 0, full_width * full_height);

     /* calculate foreground/background pixel values for ARGB */
     colors[0] = (pCursor->backRed   & 0xff00) << 8 |
                 (pCursor->backGreen & 0xff00)      |
                 (pCursor->backBlue  & 0xff00) >> 8;
     colors[1] = (pCursor->foreRed   & 0xff00) << 8 |
                 (pCursor->foreGreen & 0xff00)      |
                 (pCursor->foreBlue  & 0xff00) >> 8;

     /* take care of the bitmap scanline pad */
     cursor_pitch = BitmapBytePad(width);

     /* point to shadow data with an offset */
     shadow_dst = shadow_data + 2 + 2 * full_width;

     /* render shadow first */
     for (y=0; y<height; y++) {
          /* calculate row pointers */
          __u8 *s = pCursor->bits->source + y * cursor_pitch;
          __u8 *m = pCursor->bits->mask   + y * cursor_pitch;

          /* 8 pixels at once */
          __u8 S = 0;
          __u8 M = 0;

          for (x=0; x<width; x++) {
               /* each 8 pixels... */
               if (x % 8 == 0) {
                    /* calculate byte index */
                    int q = x / 8;

                    /* read source and mask */
                    S = s[q];
                    M = m[q];
               }

               /* set the alpha channel depending on the mask bit
                  and the color depending on the source bit */
#if BITMAP_BIT_ORDER == LSBFirst
               if (M & 1)
                    shadow_dst[x] = 0xcc;

               /* have the next bit rightmost */
               S >>= 1;
               M >>= 1;
#elif BITMAP_BIT_ORDER == MSBFirst
               if (M & 0x80)
                    shadow_dst[x] = 0xcc;

               /* have the next bit leftmost */
               S <<= 1;
               M <<= 1;
#endif
          }

          /* next line */
          shadow_dst += full_width;
     }

     /* point to cursor image */
     dst = data;

     /* blur shadow */
     for (y=0; y<full_height; y++) {
          for (x=0; x<full_width; x++) {
               int alpha = 0;

               alpha += SHADOW_XY( x, y );
               alpha += SHADOW_XY( x, y - 1 );
               alpha += SHADOW_XY( x, y + 1 );
               alpha += SHADOW_XY( x - 2, y );
               alpha += SHADOW_XY( x - 1, y );
               alpha += SHADOW_XY( x + 1, y );

               dst[x] = ((alpha / 6) << 24) | 0x303030;
          }

          /* next line */
          dst += pitch / 4;
     }

     /* point to cursor image */
     dst = data;

     /* render cursor image */
     for (y=0; y<height; y++) {
          /* calculate row pointers */
          __u8 *s = pCursor->bits->source + y * cursor_pitch;
          __u8 *m = pCursor->bits->mask   + y * cursor_pitch;

          /* 8 pixels at once */
          __u8 S = 0;
          __u8 M = 0;

          for (x=0; x<width; x++) {
               /* each 8 pixels... */
               if (x % 8 == 0) {
                    /* calculate byte index */
                    int q = x / 8;

                    /* read source and mask */
                    S = s[q];
                    M = m[q];
               }

               /* set the alpha channel depending on the mask bit
                  and the color depending on the source bit */
#if BITMAP_BIT_ORDER == LSBFirst
               if (M & 1)
                    dst[x] = 0xff000000 | colors[S & 1];

               /* have the next bit rightmost */
               S >>= 1;
               M >>= 1;
#elif BITMAP_BIT_ORDER == MSBFirst
               if (M & 0x80)
                    dst[x] = 0xff000000 | colors[S >> 7];

               /* have the next bit leftmost */
               S <<= 1;
               M <<= 1;
#endif
          }

          /* next line */
          dst += pitch / 4;
     }

     /* unlock before returning */
     surface->Unlock (surface);

     return surface;
}

/*
===========================================================================

 Pointer sprite functions

===========================================================================
*/

/*
 * DirectFBRealizeCursor
 * Convert the X cursor representation to an ARGB DirectFB surface.
 */
static Bool
XDirectFBRealizeCursor (ScreenPtr pScreen,
                        CursorPtr pCursor)
{
     if (!pCursor || !pCursor->bits)
          return FALSE;

     return TRUE;
}


/*
 * DirectFBUnrealizeCursor
 * Free the storage space associated with a realized cursor.
 */
static Bool
XDirectFBUnrealizeCursor (ScreenPtr pScreen,
                          CursorPtr pCursor)
{
     return TRUE;
}


/*
 * DirectFBSetCursor
 * Set the cursor sprite and position.
 */
static void
XDirectFBSetCursor(ScreenPtr pScreen,
                   CursorPtr pCursor,
                   int       x,
                   int       y)
{
     DirectFBScreenPtr      ScreenPriv = DIRECTFB_PRIV(pScreen);
     IDirectFBDisplayLayer *layer      = ScreenPriv->layer;

     if (pCursor) {
          IDirectFBSurface *surface;

          if (!pCursor->bits)
               return;

          /* make new cursor image */
          surface = MakeCursorSurface(pCursor);
          if (!surface)
               return;

          /* set the cursor shape */
          XDirectFBSetCursorShape (surface,
                                   pCursor->bits->xhot,
                                   pCursor->bits->yhot);

          /* show the cursor */
          layer->SetCursorOpacity (layer, 0xff);
          layer->EnableCursor (layer, 1);

          /* destroy the cursor image */
          surface->Release (surface);
     }
     else {
          /* hide the cursor */

          /* NOTE: we still need cursor events, so we must not disable the cursor,
             otherwise it won't be enabled again because the cursor cannot move
             away from the window that disabled it */

          layer->SetCursorOpacity (layer, 0);
     }
}


/*
 * DirectFBMoveCursor
 * Move the cursor. This is a noop for XDirectFB.
 */
static void
XDirectFBMoveCursor (ScreenPtr   pScreen,
                     int         x,
                     int         y)
{
}


static miPointerSpriteFuncRec directfbSpriteFuncsRec = {
     XDirectFBRealizeCursor,
     XDirectFBUnrealizeCursor,
     XDirectFBSetCursor,
     XDirectFBMoveCursor
};

/*
===========================================================================

 Pointer screen functions

===========================================================================
*/

/*
 * DirectFBCursorOffScreen
 */
static Bool
XDirectFBCursorOffScreen (ScreenPtr *pScreen,
                          int       *x,
                          int       *y)
{
     return FALSE;
}


/*
 * DirectFBCrossScreen
 */
static void
XDirectFBCrossScreen (ScreenPtr pScreen,
                      Bool      entering)
{
}


/*
 * DirectFBWarpCursor
 *  Change the cursor position without generating an event or motion history.
 *  The input coordinates (x,y) are in pScreen-local X11 coordinates.
 *
 */
static void
XDirectFBWarpCursor (ScreenPtr pScreen,
                     int       x,
                     int       y)
{
     static Bool            neverMoved = TRUE;
     DirectFBScreenPtr      ScreenPriv = DIRECTFB_PRIV(pScreen);
     IDirectFBDisplayLayer *layer      = ScreenPriv->layer;

     if (neverMoved) {
          /* Don't move the cursor the first time. This is the jump-to-center
             initialization, and it's annoying because we may still be in MacOS. */
          neverMoved = FALSE;
          return;
     }

     layer->WarpCursor (layer,
                        x + dixScreenOrigins[pScreen->myNum].x + directfbMainScreenX,
                        y + dixScreenOrigins[pScreen->myNum].y + directfbMainScreenY);

     miPointerWarpCursor(pScreen, x, y);
}


static miPointerScreenFuncRec directfbScreenFuncsRec = {
     XDirectFBCursorOffScreen,
     XDirectFBCrossScreen,
     XDirectFBWarpCursor,
};

/*
===========================================================================

 Other screen functions

===========================================================================
*/

/*
 * XDirectFBCursorQueryBestSize
 * Handle queries for best cursor size
 */
static void
XDirectFBCursorQueryBestSize (int             class,
                              unsigned short *width,
                              unsigned short *height,
                              ScreenPtr       pScreen)
{
     DirectFBCursorScreenPtr ScreenPriv = CURSOR_PRIV(pScreen);

     if (class == CursorShape) {
          /* what is a reasonable limit? */
          *width  = 1024;
          *height = 1024;
     }
     else
          (*ScreenPriv->QueryBestSize)(class, width, height, pScreen);
}


/*
 * XDirectFBInitCursor
 * Initialize cursor support
 */
Bool
XDirectFBInitCursor (ScreenPtr pScreen)
{
     DirectFBCursorScreenPtr ScreenPriv;
     miPointerScreenPtr      PointPriv;

     /* initialize software cursor handling (always needed as backup) */
     if (!miDCInitialize(pScreen, &directfbScreenFuncsRec)) {
          return FALSE;
     }

     /* allocate private storage for this screen's DirectFB cursor info */
     if (directfbCursorGeneration != serverGeneration) {
          if ((directfbCursorScreenIndex = AllocateScreenPrivateIndex()) < 0)
               return FALSE;

          directfbCursorGeneration = serverGeneration;
     }

     ScreenPriv = xcalloc( 1, sizeof(DirectFBCursorScreenRec) );
     if (!ScreenPriv)
          return FALSE;

     CURSOR_PRIV(pScreen) = ScreenPriv;

     /* override some screen procedures */
     ScreenPriv->QueryBestSize = pScreen->QueryBestSize;
     pScreen->QueryBestSize = XDirectFBCursorQueryBestSize;

     PointPriv = (miPointerScreenPtr)
                 pScreen->devPrivates[miPointerScreenIndex].ptr;

     ScreenPriv->spriteFuncs = PointPriv->spriteFuncs;
     PointPriv->spriteFuncs = &directfbSpriteFuncsRec;

     return TRUE;
}

