/*
  FTDB database system
  Copyright (C) 1995 Erik Troan, North Carolina State University
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "tcl.h"
#include "ftdb.h"
#include "ftdb_tcl.h"

struct DBInfo
{
    FullTextDB db;
    char ** Fields;
    int NumFields;
} ;

static Tcl_HashTable dbtable;
static Tcl_HashTable searchtable;

int FTDB_tcl_open(ClientData clientData, Tcl_Interp * interp, int argc, 
		  char ** argv)
{
    static int count = 0;
    struct DBInfo * dbi;
    int isnew;
    char * rc;
    Tcl_HashEntry * Entry;
    int ReadOnly = 0;
        
    if (argc != 2 && argc != 3)
    {
	interp->result = "one or two argument expected to ftdb_open";
	return TCL_ERROR;
    }
    
    if (argc == 3)
    {
	if (!strcmp(argv[2], "readonly"))
	    ReadOnly = 1;
	else if (!strcmp(argv[2], "readwrite"))
	    ReadOnly = 0;
	else
	{
	    interp->result = "Database may be opened readonly or readwrite";
	    return TCL_ERROR;
	}
    }

    dbi = malloc(sizeof(*dbi));
        
    if (!count)
    {
	Tcl_InitHashTable(&dbtable, TCL_STRING_KEYS);
    }
    
    rc = OpenFTDB(argv[1], &dbi->db, ReadOnly);
    if (rc)
    {
	interp->result = rc;
	return TCL_ERROR;
    }
    
    sprintf(interp->result, "ftdb%d", count++);

    Entry = Tcl_CreateHashEntry(&dbtable, interp->result, &isnew);
    Tcl_SetHashValue(Entry, dbi);
    
    dbi->Fields = malloc(1);
    dbi->NumFields = 0;

    return TCL_OK;
}

int FTDB_tcl_define(ClientData clientData, Tcl_Interp * interp, int argc, 
		    char ** argv)
{
    Tcl_HashEntry * Entry;
    struct DBInfo * dbi;
    char ** FieldList;
    int NumFields;
    int i, j;
    char ** FieldInfo;
    int NumFieldInfo;
    char * rc;
    int KeepIndex;
    int StoreData;
    int WriteCache;

    if (argc != 3)
    {
	interp->result = "two arguments expected to ftdb_define";
	return TCL_ERROR;
    }

    Entry = Tcl_FindHashEntry(&dbtable, argv[1]);
    if (!Entry) 
    {
	interp->result = "bad ftdb database";
	return TCL_ERROR;
    }

    dbi = Tcl_GetHashValue(Entry);

    if (Tcl_SplitList(interp, argv[2], &NumFields, &FieldList) != TCL_OK)
    {
	return TCL_ERROR;
    }
    
    for (i = 0; i < NumFields; i++)
    {
	if (Tcl_SplitList(interp, FieldList[i], &NumFieldInfo,
			  &FieldInfo) != TCL_OK)
	{
	    return TCL_ERROR;
	}
	
	if (NumFieldInfo < 2 || NumFieldInfo > 5)
	{
	    interp->result = "field definition: fieldname <type> <noindex|nostore|writecache>";
	    return TCL_ERROR;
	}
	
	if (strcmp(FieldInfo[1], "string"))
	{
	    interp->result = "only string fields are currently supported";
	    return TCL_ERROR;
	}

 	StoreData = KeepIndex = 1;
	WriteCache = 0;

	for (j = 2; j < NumFieldInfo; j++)
	{
	    if (!strcmp(FieldInfo[j], "noindex")) KeepIndex = 0;
	    if (!strcmp(FieldInfo[j], "nostore")) StoreData = 0;
	    if (!strcmp(FieldInfo[j], "writecache")) WriteCache = 1;
	}

	rc = DefineFieldFTDB(dbi->db, FieldInfo[0], FTDB_STRING, StoreData, 
			     KeepIndex, WriteCache);
	if (rc) 
	{
	    interp->result = rc;
	    return TCL_ERROR;
	}

	dbi->Fields = realloc(dbi->Fields, (dbi->NumFields + 1) * 
			      sizeof(char *));
	if (!dbi->Fields)
	{
	    interp->result = "Out of memory";
	    return TCL_ERROR;
	}
	dbi->Fields[dbi->NumFields] = strdup(FieldInfo[0]);
	if (!dbi->Fields[dbi->NumFields])
	{
	    interp->result = "Out of memory";
	    return TCL_ERROR;
	}
	dbi->NumFields++;
	
	free(FieldInfo);
    }	
	
    free(FieldList);

    return TCL_OK;
}

int FTDB_tcl_add(ClientData clientData, Tcl_Interp * interp, int argc, 
		 char ** argv)
{
    Tcl_HashEntry * Entry;
    struct DBInfo * dbi;
    int i;
    char * rc;
    Record rec;

    if (argc < 3)
    {
	interp->result = "at least two arguments expected to ftdb_add";
	return TCL_ERROR;
    }

    Entry = Tcl_FindHashEntry(&dbtable, argv[1]);
    if (!Entry) 
    {
	interp->result = "bad ftdb database";
	return TCL_ERROR;
    }

    dbi = Tcl_GetHashValue(Entry);

    rc = StartRecordFTDB(dbi->db, &rec);
    if (rc)
    {
	interp->result = rc;
	return TCL_ERROR;
    }
    
    if (dbi->NumFields != (argc - 2))
    {
	interp->result = "Wrong number of fields";
	return TCL_ERROR;
    }
    
    for (i = 2; i < argc; i++)
    {
	rc = AddStringFieldToRecordFTDB(dbi->db, rec, argv[i]);
	if (rc)
	{
	    interp->result = rc;
	    return TCL_ERROR;
	}
    }
    rc = AddRecordFTDB(dbi->db, rec);

    FreeRecordFTDB(dbi->db, rec);

    if (rc)
    {
	interp->result = rc;
	return TCL_ERROR;
    }
    
    return TCL_OK;
}

int FTDB_tcl_search(ClientData clientData, Tcl_Interp * interp, int argc, 
		    char ** argv)
{
    Tcl_HashEntry * Entry;
    struct DBInfo * dbi;
    char * rc;
    MatchList Matches;
    static int count = 0;
    int isnew;
    char * FieldName;

    if (argc != 4)
    {
	interp->result = "three arguments expected to ftdb_search";
	return TCL_ERROR;
    }

    Entry = Tcl_FindHashEntry(&dbtable, argv[1]);
    if (!Entry) 
    {
	interp->result = "bad ftdb database";
	return TCL_ERROR;
    }

    dbi = Tcl_GetHashValue(Entry);

    if (!strcmp(argv[2], "-any"))
    {
	FieldName = NULL;
    }
    else
    {
	FieldName = argv[3];
    }

    rc = SearchFTDB(dbi->db, argv[3], FieldName, &Matches);
    if (rc)
    {
	interp->result = rc;
	return TCL_ERROR;
    }

    if (!count)
    {
	Tcl_InitHashTable(&searchtable, TCL_STRING_KEYS);
    }

    sprintf(interp->result, "ftdb_s%d", count++);

    Entry = Tcl_CreateHashEntry(&searchtable, interp->result, &isnew);
    Tcl_SetHashValue(Entry, Matches);
    
    return 0;
}

int FTDB_tcl_getreclist(ClientData clientData, Tcl_Interp * interp, int argc, 
		     char ** argv)
{
    MatchList Match;
    char num[50];
    Tcl_HashEntry * Entry;
        
    if (argc != 2)
    {
	interp->result = "ftdb_getreclist <db>";
	return TCL_ERROR;
    }

    Entry = Tcl_FindHashEntry(&searchtable, argv[1]);
    if (!Entry) 
    {
	interp->result = "bad ftdb search result";
	return TCL_ERROR;
    }
    Match = Tcl_GetHashValue(Entry);
    
    while (Match)
    {
	sprintf(num, "%u", Match->Offset);
	Tcl_AppendElement(interp, num);
	Match = Match->Next;
    }

    return TCL_OK;
}

int FTDB_tcl_getrecord(ClientData clientData, Tcl_Interp * interp, int argc, 
		       char ** argv)
{
    Tcl_HashEntry * Entry;
    struct DBInfo * dbi;
    char * rc;
    int Offset;
    Record rec;
    int i;

    if (argc != 4)
    {
	interp->result = "three arguments expected to ftdb_getrecord";
	return TCL_ERROR;
    }

    Entry = Tcl_FindHashEntry(&dbtable, argv[1]);
    if (!Entry) 
    {
	interp->result = "bad ftdb database";
	return TCL_ERROR;
    }

    dbi = Tcl_GetHashValue(Entry);
    
    /**** This needs to understand unsigned, but it's easy */
    if (Tcl_GetInt(interp, argv[2], &Offset) != TCL_OK) return TCL_ERROR;
    
    rc = GetRecordFTDB(dbi->db, Offset, &rec);

    for (i = 0; i < rec->NextFieldNum; i++)
    {
	if (Tcl_SetVar2(interp, argv[3], dbi->Fields[i], rec->Fields[i], 
			TCL_LEAVE_ERR_MSG) == NULL)
	{
	    return TCL_ERROR;
	}
    }

    interp->result = "";
    FreeRecordFTDB(dbi->db, rec);

    return TCL_OK;
}

int FTDB_tcl_close(ClientData clientData, Tcl_Interp * interp, int argc, 
		   char ** argv)
{
    Tcl_HashEntry * Entry;
    struct DBInfo * dbi;
    int i;

    if (argc != 2)
    {
	interp->result = "one argument expected to ftdb_close";
	return TCL_ERROR;
    }

    Entry = Tcl_FindHashEntry(&dbtable, argv[1]);
    if (!Entry) 
    {
	interp->result = "bad ftdb database";
	return TCL_ERROR;
    }

    dbi = Tcl_GetHashValue(Entry);

    CloseFTDB(dbi->db);

    for (i = 0; i < dbi->NumFields; i++) free(dbi->Fields[i]);
    free(dbi->Fields);
    free(dbi);

    Tcl_DeleteHashEntry(Entry);
    
    return TCL_OK;
}

int FTDB_tcl_intersect(ClientData clientData, Tcl_Interp * interp, int argc, 
		       char ** argv)
{
    MatchList One, Two, Result;
    Tcl_HashEntry * EntryOne, * EntryTwo;
    char * rc;
        
    if (argc != 3)
    {
	interp->result = "ftcb_intesect <reclist> <reclist> : reclist";
	return TCL_ERROR;
    }

    EntryOne = Tcl_FindHashEntry(&searchtable, argv[1]);
    if (!EntryOne) 
    {
	interp->result = "bad ftdb search result";
	return TCL_ERROR;
    }
    One = Tcl_GetHashValue(EntryOne);
    
    EntryTwo = Tcl_FindHashEntry(&searchtable, argv[2]);
    if (!EntryTwo) 
    {
	interp->result = "bad ftdb search result";
	return TCL_ERROR;
    }
    Two = Tcl_GetHashValue(EntryTwo);

    rc = IntersectMatchesFTDB(One, Two, &Result);
    if (!EntryTwo) 
    {
	interp->result = rc;
	return TCL_ERROR;
    }

    Tcl_SetHashValue(EntryTwo, Result);
    interp->result = strdup(argv[2]);
    interp->freeProc = (Tcl_FreeProc * ) free;
    Tcl_DeleteHashEntry(EntryOne);
    
    return TCL_OK;
}

int FTDB_tcl_union(ClientData clientData, Tcl_Interp * interp, int argc, 
		   char ** argv)
{
    MatchList One, Two, Result;
    Tcl_HashEntry * EntryOne, * EntryTwo;
    char * rc;
        
    if (argc != 3)
    {
	interp->result = "two arguments expected to ftdb_union";
	return TCL_ERROR;
    }

    EntryOne = Tcl_FindHashEntry(&searchtable, argv[1]);
    if (!EntryOne) 
    {
	interp->result = "bad ftdb search result";
	return TCL_ERROR;
    }
    One = Tcl_GetHashValue(EntryOne);
    
    EntryTwo = Tcl_FindHashEntry(&searchtable, argv[2]);
    if (!EntryTwo) 
    {
	interp->result = "bad ftdb search result";
	return TCL_ERROR;
    }
    Two = Tcl_GetHashValue(EntryTwo);

    rc = UnionMatchesFTDB(One, Two, &Result);
    if (!EntryTwo) 
    {
	interp->result = rc;
	return TCL_ERROR;
    }

    Tcl_SetHashValue(EntryTwo, Result);
    interp->result = strdup(argv[2]);
    interp->freeProc = (Tcl_FreeProc * ) free;
    Tcl_DeleteHashEntry(EntryOne);
    
    return TCL_OK;
}

int FTDB_tcl_reindex(ClientData clientData, Tcl_Interp * interp, int argc, 
		     char ** argv)
{
    Tcl_HashEntry * Entry;
    struct DBInfo * dbi;

    if (argc != 2)
    {
	interp->result = "one argument expected to ftdb_reindex";
	return TCL_ERROR;
    }

    Entry = Tcl_FindHashEntry(&dbtable, argv[1]);
    if (!Entry) 
    {
	interp->result = "bad ftdb database";
	return TCL_ERROR;
    }

    dbi = Tcl_GetHashValue(Entry);

    interp->result = ReindexFTDB(dbi->db);
    if (interp->result) return TCL_ERROR;

    interp->result = "";
    
    return TCL_OK;
}

int Tclftdb_Init(Tcl_Interp * interp)
{
    Tcl_CreateCommand(interp, "ftdb_open", FTDB_tcl_open, NULL, NULL);
    Tcl_CreateCommand(interp, "ftdb_define", FTDB_tcl_define, NULL, NULL);
    Tcl_CreateCommand(interp, "ftdb_add", FTDB_tcl_add, NULL, NULL);
    Tcl_CreateCommand(interp, "ftdb_search", FTDB_tcl_search, NULL, NULL);
    Tcl_CreateCommand(interp, "ftdb_getreclist", FTDB_tcl_getreclist, NULL, 
		      NULL);
    Tcl_CreateCommand(interp, "ftdb_getrecord", FTDB_tcl_getrecord,NULL, NULL);
    Tcl_CreateCommand(interp, "ftdb_close", FTDB_tcl_close, NULL, NULL);
    Tcl_CreateCommand(interp, "ftdb_intersect", FTDB_tcl_intersect, NULL, 
		      NULL);
    Tcl_CreateCommand(interp, "ftdb_union", FTDB_tcl_union, NULL, NULL);
    Tcl_CreateCommand(interp, "ftdb_reindex", FTDB_tcl_reindex, NULL, NULL);
   
    return TCL_OK;
}

