#include "cupsparser.h"

#include <qtimer.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qapplication.h>
#include <kconfig.h>
#include <kglobal.h>
#include <kdebug.h>
#include <klocale.h>

CupsParser::CupsParser( const QString& name, const QString& filename, QObject *parent )
	: QObject( parent ), m_name( name ), m_filename( filename )
{
	m_timer = new QTimer( this );
	connect( m_timer, SIGNAL( timeout() ), SLOT( slotCheck() ) );

	m_filterenabled = false;
	m_seekpos = 0;
	m_parseinterval = 2000;

	restoreParser();
}

CupsParser::~CupsParser()
{
	saveParser();
}

void CupsParser::restoreParser()
{
	kdDebug() << "Restoring parser: " << parserName() << endl;
	KConfig *conf = KGlobal::config();
	conf->setGroup( parserName() );
	QDateTime dummy;
	m_start = conf->readNumEntry( "Start", 1000 );
	setFilter( conf->readDateTimeEntry( "From", &dummy ), conf->readDateTimeEntry( "To", &dummy ) );
	setFilter( conf->readNumEntry( "Field", -1 ), QRegExp( conf->readEntry( "Match", QString::null ) ) );
	m_filename = conf->readEntry( "File", m_filename );
}

void CupsParser::saveParser()
{
	kdDebug() << "Saving parser: " << parserName() << endl;
	KConfig *conf = KGlobal::config();
	conf->setGroup( parserName() );
	conf->writeEntry( "From", m_filterfrom );
	conf->writeEntry( "To", m_filterto );
	conf->writeEntry( "Start", m_start );
	conf->writeEntry( "Field", m_filterfield );
	conf->writeEntry( "Match", m_filtermatch.pattern() );
	conf->writeEntry( "File", m_filename );
}

void CupsParser::setParseInterval( unsigned int t )
{
	m_parseinterval = t;
}

unsigned int CupsParser::parseInterval() const
{
	return m_parseinterval;
}

void CupsParser::setFileName( const QString& f )
{
	m_filename = f;
	m_seekpos = 0;
	m_lastcheck = QDateTime();
	m_timer->stop();
}

void CupsParser::startParsing( bool flag )
{
	m_timer->stop();
	initSeekPos( m_start );
	emit restarted();
	m_lastcheck = QDateTime();
	if ( flag )
		slotCheck();
	else
		m_timer->start( 1, true );
}

void CupsParser::stopParsing()
{
	m_timer->stop();
}

void CupsParser::resumeParsing()
{
	m_timer->start( m_parseinterval, true );
}

void CupsParser::setStart( int n )
{
	m_start = n;
}

int CupsParser::start() const
{
	return m_start;
}

int CupsParser::filterField() const
{
	return m_filterfield;
}

QString CupsParser::filterMatch() const
{
	return m_filtermatch.pattern();
}

QDateTime CupsParser::filterFrom() const
{
	return m_filterfrom;
}

QDateTime CupsParser::filterTo() const
{
	return m_filterto;
}

void CupsParser::initSeekPos( int n )
{
	m_seekpos = 0;
	if ( n >= 0 )
	{
		QFile file( m_filename );
		if ( file.exists() && file.open( IO_ReadOnly ) )
		{
			char c = 0;

			file.at( file.size() );
			while ( n >= 0 && c != -1 )
			{
				do
				{
					if ( file.at() == 0 )
						c = -1;
					else
					{
						file.at( file.at()-1 );
						c = file.getch();
						file.at( file.at()-1 );
					}
				} while ( c != '\n' && c != -1 );
				n--;
			}
			m_seekpos = ( c == -1 ? 0 : file.at()+1 );
			file.close();
		}
	}
}

QString CupsParser::fileName() const
{
	return m_filename;
}

void CupsParser::slotCheck()
{
	if ( !m_filename.isEmpty() )
	{
		QFileInfo fi( m_filename );
		if ( !m_lastcheck.isValid() || fi.lastModified() > m_lastcheck )
		{
			QFile f( m_filename );
			if ( !f.exists() )
				error( i18n( "Log file does not exist: %1." ).arg( m_filename ) );
			else if ( !f.open( IO_ReadOnly ) )
				error( i18n( "Unable to open log file for reading: %1. Permission denied." ).arg( m_filename ) );
			else if ( !f.at( m_seekpos ) )
				error( i18n( "Internal error: unable to set file offset correctly." ) );
			else
			{
				QTextStream t( &f );
				parse( t );
				m_seekpos = f.at();
				f.close();
				m_lastcheck = fi.lastModified();
				m_timer->start( m_parseinterval, true );
			}
		}
	}
}

void CupsParser::parse( QTextStream& t )
{
	QString line;
	ItemList list;

	list.setAutoDelete( false );
	while ( !t.atEnd() )
	{
		line = t.readLine();
		Item *item = lineToItem( line );
		if ( item && item->date.isValid() && ( !m_filterenabled || filter( item ) ) )
			list.append( item );
		else
			delete item;

		if ( list.count() >= 1000 || t.atEnd() )
		{
			emit newItems( list );
			list.clear();
			qApp->processEvents();
		}
	}
}

void CupsParser::setFilter( const QDateTime& from, const QDateTime& to )
{
	m_filterfrom = from;
	m_filterto = to;
	m_filterenabled = ( m_filterfrom.isValid() || m_filterto.isValid() ||
			( m_filterfield != -1 && m_filtermatch.isValid() && !m_filtermatch.isEmpty() ) );
}

void CupsParser::setFilter( int col, const QRegExp& match )
{
	m_filterfield = col;
	m_filtermatch = match;
	m_filterenabled = ( m_filterfrom.isValid() || m_filterto.isValid() ||
			( m_filterfield != -1 && m_filtermatch.isValid() && !m_filtermatch.isEmpty() ) );
}

bool CupsParser::filter( CupsParser::Item *item )
{
	if ( ( m_filterfrom.isValid() && m_filterfrom > item->date ) ||
		 ( m_filterto.isValid() && item->date > m_filterto ) )
		return false;
	if ( m_filterfield != -1 && !m_filtermatch.exactMatch( item->fields[ m_filterfield ].asString() ) )
		return false;
	return true;
}

QString CupsParser::parserName() const
{
	return m_name;
}

QStringList CupsParser::parsers()
{
	return QStringList() << "AccessParser" << "ErrorParser" << "PageParser";
}

void CupsParser::setupRMB( QListViewItem*, QPopupMenu* )
{
}

static const char *months[ 12 ] =
{
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

void CupsParser::Item::parseDate( const QString& str )
{
	int d, y, h, m, s, mo;
	char mon[4] = {0};

	sscanf( str.ascii(), "%02d/%03s/%04d:%02d:%02d:%02d", &d, mon, &y, &h, &m, &s );
	for ( mo = 0; mo < 12; mo++ )
		if ( strcmp( months[ mo ], mon ) == 0 )
			break;
	date = QDateTime( QDate( y, mo+1, d ), QTime( h, m, s ) );
}

#include "cupsparser.moc"
