/***************************************************************************
                          kscribble.cpp  -  description
                             -------------------
    begin                : Mon Jan 31 11:05:05 CET 2000
    copyright            : (C) 2000 by Ralf Nolden
    email                : Ralf.Nolden@post.rwth-aachen.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// include files for QT
#include <qdir.h>
#include <qprinter.h>
#include <qvbox.h>
#include <qwhatsthis.h>
#include <qtooltip.h>
#include <qtoolbutton.h>
#include <qimage.h>
#include <qdragobject.h>


// include files for KDE
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kcolordlg.h>
#include <kmenubar.h>
#include <klocale.h>
#include <kconfig.h>

// application specific includes
#include "kscribble.h"
#include "kscribbleview.h"
#include "kscribbledoc.h"
#include "resource.h"
#include "kpenbrushdlg.h"


KScribbleApp::KScribbleApp()
{
  config=kapp->config();
  printer = new QPrinter;
	untitledCount=0;
  pDocList = new QList<KScribbleDoc>();
  pDocList->setAutoDelete(true);
  setAcceptDrops(true);

  ///////////////////////////////////////////////////////////////////
  // call inits to invoke all other construction parts
  initMenuBar();
  initToolBar();
  initStatusBar();
  initKeyAccel();
  initView();
	
  readOptions();

  ///////////////////////////////////////////////////////////////////
  // disable menu and toolbar items at startup
  disableCommand(ID_EDIT_UNDO);
}

KScribbleApp::~KScribbleApp()
{
  delete printer;
}

void KScribbleApp::initKeyAccel()
{
  keyAccel = new KAccel(this);
	
  // fileMenu accelerators
  keyAccel->connectItem(KStdAccel::New, this, SLOT(slotFileNew()));
  keyAccel->connectItem(KStdAccel::Open, this, SLOT(slotFileOpen()));
  keyAccel->connectItem(KStdAccel::Save, this, SLOT(slotFileSave()));
  keyAccel->connectItem(KStdAccel::Close, this, SLOT(slotFileClose()));
  keyAccel->connectItem(KStdAccel::Print, this, SLOT(slotFilePrint()));
  keyAccel->connectItem(KStdAccel::Quit, this, SLOT(slotFileQuit()));
  // editMenu accelerators
  keyAccel->connectItem(KStdAccel::Cut, this, SLOT(slotEditCut()));
  keyAccel->connectItem(KStdAccel::Copy, this, SLOT(slotEditCopy()));
  keyAccel->connectItem(KStdAccel::Paste, this, SLOT(slotEditPaste()));

  keyAccel->connectItem(KStdAccel::Help, this, SLOT(appHelpActivated()));
			
  keyAccel->changeMenuAccel(pFileMenu, ID_FILE_NEW, KStdAccel::New);
  keyAccel->changeMenuAccel(pFileMenu, ID_FILE_OPEN, KStdAccel::Open);
  keyAccel->changeMenuAccel(pFileMenu, ID_FILE_SAVE, KStdAccel::Save);
  keyAccel->changeMenuAccel(pFileMenu, ID_FILE_CLOSE, KStdAccel::Close);
  keyAccel->changeMenuAccel(pFileMenu, ID_FILE_PRINT, KStdAccel::Print);
  keyAccel->changeMenuAccel(pFileMenu, ID_FILE_QUIT, KStdAccel::Quit);

  keyAccel->changeMenuAccel(pEditMenu, ID_EDIT_CUT, KStdAccel::Cut);
  keyAccel->changeMenuAccel(pEditMenu, ID_EDIT_COPY, KStdAccel::Copy);
  keyAccel->changeMenuAccel(pEditMenu, ID_EDIT_PASTE, KStdAccel::Paste);

  keyAccel->readSettings();	
}

void KScribbleApp::initMenuBar()
{
  ///////////////////////////////////////////////////////////////////
  // MENUBAR
  pRecentFileMenu = new QPopupMenu(this);
  connect(pRecentFileMenu, SIGNAL(activated(int)), SLOT(slotFileOpenRecent(int)));

  ///////////////////////////////////////////////////////////////////
  // menuBar entry file-Menu
  pFileMenu = new QPopupMenu(this);
  pFileMenu->insertItem(BarIcon("filenew"), i18n("&New"), ID_FILE_NEW);
  pFileMenu->insertItem(BarIcon("fileopen"), i18n("&Open..."), ID_FILE_OPEN);
  pFileMenu->insertItem(i18n("Open &recent"), pRecentFileMenu, ID_FILE_OPEN_RECENT);

  pFileMenu->insertItem(i18n("&Close"), ID_FILE_CLOSE);
  pFileMenu->insertSeparator();
  pFileMenu->insertItem(BarIcon("filefloppy") ,i18n("&Save"), ID_FILE_SAVE);
  pFileMenu->insertItem(i18n("Save &As..."), ID_FILE_SAVE_AS);
  pFileMenu->insertSeparator();
  pFileMenu->insertItem(BarIcon("fileprint"), i18n("&Print..."), ID_FILE_PRINT);
  pFileMenu->insertSeparator();
  pFileMenu->insertItem(i18n("E&xit"), ID_FILE_QUIT);
	
  ///////////////////////////////////////////////////////////////////
  // menuBar entry edit-Menu
  pEditMenu = new QPopupMenu(this);
  pEditMenu->insertItem(BarIcon("undo"), i18n("&Undo"), ID_EDIT_UNDO);
  pEditMenu->insertSeparator();
  pEditMenu->insertItem(BarIcon("editcut"), i18n("Cu&t"), ID_EDIT_CUT);
  pEditMenu->insertItem(BarIcon("editcopy"), i18n("&Copy"), ID_EDIT_COPY);
  pEditMenu->insertItem(BarIcon("editpaste"), i18n("&Paste"), ID_EDIT_PASTE);
  pEditMenu->insertItem(BarIcon("delete"),i18n("&Clear All"), ID_EDIT_CLEAR_ALL);

  ///////////////////////////////////////////////////////////////////
  // menuBar entry pen-Menu
  pPenMenu = new QPopupMenu();
  pPenMenu->insertItem(i18n("&Color"), ID_PEN_COLOR);
  pPenMenu->insertItem(i18n("&Brush"), ID_PEN_BRUSH);

  ///////////////////////////////////////////////////////////////////
  // menuBar entry view-Menu
  pViewMenu = new QPopupMenu(this);
  pViewMenu->setCheckable(true);
  pViewMenu->insertItem(i18n("&Toolbar"), ID_VIEW_TOOLBAR);
  pViewMenu->insertItem(i18n("&Statusbar"), ID_VIEW_STATUSBAR);

  ///////////////////////////////////////////////////////////////////
  // menuBar entry window-Menu
  pWindowMenu = new QPopupMenu(this);
  pWindowMenu->setCheckable(true);


  ///////////////////////////////////////////////////////////////////
  // menuBar entry helpMenu
  QPopupMenu* pHelpMenu = helpMenu(i18n("KScribble" VERSION "\n\n(c) 2000 by\nRalf Nolden\nRalf.Nolden@post.rwth-aachen.de"));

  ///////////////////////////////////////////////////////////////////
  // MENUBAR CONFIGURATION
  // insert your popup menus with the according menubar entries in the order
  // they will appear later from left to right
  menuBar()->insertItem(i18n("&File"), pFileMenu);
  menuBar()->insertItem(i18n("&Edit"), pEditMenu);
  menuBar()->insertItem(i18n("&Pen"), pPenMenu);
  menuBar()->insertItem(i18n("&View"), pViewMenu);
  menuBar()->insertItem(i18n("&Window"), pWindowMenu );
  menuBar()->insertItem(i18n("&Help"), pHelpMenu);

  ///////////////////////////////////////////////////////////////////
  // CONNECT THE MENU SLOTS WITH SIGNALS
  // for execution slots and statusbar messages
  connect(pFileMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
  connect(pFileMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));

  connect(pEditMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
  connect(pEditMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));

  connect(pPenMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
  connect(pPenMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));

  connect(pViewMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
  connect(pViewMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));

  connect(pWindowMenu, SIGNAL(aboutToShow() ), SLOT( windowMenuAboutToShow() ) );
  connect(pWindowMenu, SIGNAL(activated(int)), SLOT(commandCallback(int)));
  connect(pWindowMenu, SIGNAL(highlighted(int)), SLOT(statusCallback(int)));

}

void KScribbleApp::initToolBar()
{

  ///////////////////////////////////////////////////////////////////
  // TOOLBAR
  toolBar()->insertButton(BarIcon("filenew"), ID_FILE_NEW, true, i18n("New File"));
  toolBar()->insertButton(BarIcon("fileopen"), ID_FILE_OPEN, true, i18n("Open File"));
  toolBar()->insertButton(BarIcon("filefloppy"), ID_FILE_SAVE, true, i18n("Save File"));
  toolBar()->insertButton(BarIcon("fileprint"), ID_FILE_PRINT, true, i18n("Print"));
  toolBar()->insertSeparator();
  toolBar()->insertButton(BarIcon("editcut"), ID_EDIT_CUT, true, i18n("Cut"));
  toolBar()->insertButton(BarIcon("editcopy"), ID_EDIT_COPY, true, i18n("Copy"));
  toolBar()->insertButton(BarIcon("editpaste"), ID_EDIT_PASTE, true, i18n("Paste"));
  toolBar()->insertSeparator();
  toolBar()->insertButton(BarIcon("pencolor"), ID_PEN_COLOR, true, i18n("Color") );
  toolBar()->insertButton(BarIcon("penwidth"), ID_PEN_BRUSH, true, i18n("Width") );
  toolBar()->insertSeparator();
  toolBar()->insertButton(BarIcon("help"), ID_HELP_CONTENTS, SIGNAL(clicked()),
  				this, SLOT(appHelpActivated()), true,i18n("Help"));

  QToolButton *btnwhat = QWhatsThis::whatsThisButton(toolBar());
  QToolTip::add(btnwhat, i18n("What's this...?"));
  toolBar()->insertWidget(ID_HELP_WHATS_THIS, btnwhat->sizeHint().width(), btnwhat);

  ///////////////////////////////////////////////////////////////////
  // INSERT YOUR APPLICATION SPECIFIC TOOLBARS HERE WITH toolBar(n)


  ///////////////////////////////////////////////////////////////////
  // CONNECT THE TOOLBAR SLOTS WITH SIGNALS - add new created toolbars by their according number
  // connect for invoking the slot actions
  connect(toolBar(), SIGNAL(clicked(int)), SLOT(commandCallback(int)));
  // connect for the status help on holing icons pressed with the mouse button
  connect(toolBar(), SIGNAL(pressed(int)), SLOT(statusCallback(int)));

}

void KScribbleApp::initStatusBar()
{
  ///////////////////////////////////////////////////////////////////
  // STATUSBAR
  // TODO: add your own items you need for displaying current application status.
  statusBar()->insertItem(i18n("Ready."), ID_STATUS_MSG);
}


void KScribbleApp::initView()
{ 
  ////////////////////////////////////////////////////////////////////
  // here the main view of the KTMainWindow is created by a background box and
  // the QWorkspace instance for MDI view.
  QVBox* view_back = new QVBox( this );
  view_back->setFrameStyle( QFrame::StyledPanel | QFrame::Sunken );
  pWorkspace = new QWorkspace( view_back );
  connect(pWorkspace, SIGNAL(windowActivated(QWidget*)), this, SLOT(setWndTitle(QWidget*)));
  setView(view_back);
}

void KScribbleApp::setWndTitle(QWidget*){
	setCaption(pWorkspace->activeWindow()->caption());
}

void KScribbleApp::enableCommand(int id_)
{
  ///////////////////////////////////////////////////////////////////
  // enable menu and toolbar functions by their ID's
  menuBar()->setItemEnabled(id_, true);
  toolBar()->setItemEnabled(id_, true);
}

void KScribbleApp::disableCommand(int id_)
{
  ///////////////////////////////////////////////////////////////////
  // disable menu and toolbar functions by their ID's
  menuBar()->setItemEnabled(id_, false);
  toolBar()->setItemEnabled(id_, false);
}

void KScribbleApp::addRecentFile(const QString &file)
{
  if(recentFiles.contains(file))
    return; // it's already there

  if( recentFiles.count() < 5)
    recentFiles.prepend(file);
  else{
    recentFiles.remove(recentFiles.last());
    recentFiles.prepend(file);
  }

  pRecentFileMenu->clear();

  for ( int i =0 ; i < (int)recentFiles.count(); i++){
    pRecentFileMenu->insertItem(recentFiles.at(i));
  }

}

void KScribbleApp::createClient(KScribbleDoc* doc)
{
  KScribbleView* w = new KScribbleView(doc, pWorkspace,0,WDestructiveClose);
	w->installEventFilter(this);
  doc->addView(w);
	w->setIcon(kapp->miniIcon());
  if ( pWorkspace->windowList().isEmpty() ) // show the very first window in maximized mode
	  w->showMaximized();
  else
  	w->show();
}

void KScribbleApp::openDocumentFile(const char* file)
{
  slotStatusMsg(i18n("Opening file..."));
  KScribbleDoc* doc;
	// check, if document already open. If yes, set the focus to the first view
  for(doc=pDocList->first(); doc > 0; doc=pDocList->next())
  {
  	if(doc->pathName()==file)
  	{
 	  	KScribbleView* view=doc->firstView();	
 	  	view->setFocus();
 			return;
 		}
  }
  doc = new KScribbleDoc();
  pDocList->append(doc);
  doc->newDocument();
	// Creates an untitled window if file is 0	
	if(!file)
	{
		untitledCount+=1;
		QString fileName=QString(i18n("Untitled%1")).arg(untitledCount);
		doc->setPathName(fileName);
		doc->setTitle(fileName);
	}
	// Open the file
	else
  {
    QString format=QImageIO::imageFormat(file);
	  if(!doc->openDocument(file,format))
  		KMessageBox::error (this,i18n("Could not open document !"), i18n("Error !"));
		addRecentFile(file);
	}
	// create the window
  createClient(doc);

  slotStatusMsg(i18n("Ready."));
}


void KScribbleApp::saveOptions()
{	
  config->setGroup("General Options");
  config->writeEntry("Geometry", size());
  config->writeEntry("Show Toolbar", toolBar()->isVisible());
  config->writeEntry("Show Statusbar",statusBar()->isVisible());
  config->writeEntry("ToolBarPos", (int) toolBar()->barPos());
  config->writeEntry("Recent Files", recentFiles);
}


void KScribbleApp::readOptions()
{
	
  config->setGroup("General Options");

  // bar status settings
  bool bViewToolbar = config->readBoolEntry("Show Toolbar", true);
  menuBar()->setItemChecked(ID_VIEW_TOOLBAR, bViewToolbar);
  if(!bViewToolbar)
  {
     enableToolBar(KToolBar::Hide);
  }
	
  bool bViewStatusbar = config->readBoolEntry("Show Statusbar", true);
  menuBar()->setItemChecked(ID_VIEW_STATUSBAR, bViewStatusbar);
  if(!bViewStatusbar)
  {
    enableStatusBar(KStatusBar::Hide);
  }

  // bar position settings
  KToolBar::BarPosition toolBarPos;
  toolBarPos=(KToolBar::BarPosition) config->readNumEntry("ToolBarPos", KToolBar::Top);
  toolBar()->setBarPos(toolBarPos);

  // initialize the recent file list
  config->readListEntry("Recent Files",recentFiles);

  for (int i=0; i < (int) recentFiles.count(); i++)
  {
    pRecentFileMenu->insertItem(recentFiles.at(i));
  }

  QSize size=config->readSizeEntry("Geometry");
  if(!size.isEmpty())
  {
    resize(size);
  }
  else
    resize(400,350);

}

void KScribbleApp::saveProperties(KConfig *_cfg)
{

}


void KScribbleApp::readProperties(KConfig* _cfg)
{
}		

bool KScribbleApp::queryClose()
{

	QStringList saveFiles;
	KScribbleDoc* doc;
	if(pDocList->isEmpty())
		return true;

	for(doc=pDocList->first(); doc!=0;doc=pDocList->next()){
		if(doc->isModified())
			saveFiles.append(doc->title());
	}
	if(saveFiles.isEmpty())
		return true;
			
	switch (KMessageBox::questionYesNoList(this,
	    i18n("One or more documents have been modified.\nSave changes before exiting?"),saveFiles)) 		
	{
  	case KMessageBox::Yes:
			for(doc=pDocList->first(); doc!=0;doc=pDocList->next()){
			  if(doc->title().contains(i18n("Untitled")))
				  slotFileSaveAs();
	  		else
	  		{
  	  		if(!doc->saveDocument(doc->pathName())){
  		  		KMessageBox::error (this,i18n("Could not save the current document !"), i18n("I/O Error !"));
			      return false;
			    }
  	  	}
		  }
      return true;
  	case KMessageBox::No:
  	default:
      return true;
	}
}

bool KScribbleApp::queryExit()
{
  saveOptions();
  return true;
}

bool KScribbleApp::eventFilter(QObject* object, QEvent* event){
	if(event->type() == QEvent::Close)
	{
		QCloseEvent* e=(QCloseEvent*)event;
		KScribbleView* pView=(KScribbleView*)object;
		KScribbleDoc* pDoc=pView->getDocument();
		if(pDoc->canCloseFrame(pView))
		{
 			pDoc->removeView(pView);
 			if(!pDoc->firstView())
		 		pDocList->remove(pDoc);
 			
			e->accept();
  		//////////////	
  		if(pWorkspace->windowList().count()==1)
  			setPlainCaption(kapp->caption());
  		else
  			setCaption(pWorkspace->activeWindow()->caption());			
  		//////////////
		}
		else
			e->ignore();
	}
	return QWidget::eventFilter( object, event );    // standard event processing
}

/////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATION
/////////////////////////////////////////////////////////////////////


void KScribbleApp::slotFileNew()
{
  slotStatusMsg(i18n("Creating new document..."));

	openDocumentFile();

  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotFileOpen()
{
  slotStatusMsg(i18n("Opening file..."));
	
  QString fileToOpen=KFileDialog::getOpenFileName(QDir::currentDirPath(),
            KImageIO::pattern(KImageIO::Reading), this, i18n("Open File..."));
  if(!fileToOpen.isEmpty())
  {
		openDocumentFile(fileToOpen);		
  }

  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotFileOpenRecent(int id_)
{
  slotStatusMsg(i18n("Opening file..."));
  	
  openDocumentFile(pRecentFileMenu->text(id_));
	
  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotFileSave()
{
  slotStatusMsg(i18n("Saving file..."));
	KScribbleView* m = (KScribbleView*)pWorkspace->activeWindow();
	if( m )
	{
	  KScribbleDoc* doc =	m->getDocument();
	  if(doc->title().contains(i18n("Untitled")))
	   slotFileSaveAs();
	  else
  	  if(!doc->saveDocument(doc->pathName()))
  		  KMessageBox::error (this,i18n("Could not save the current document !"), i18n("I/O Error !"));
  }
	

  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotFileSaveAs()
{
  slotStatusMsg(i18n("Saving file with a new filename..."));

  QString newName=KFileDialog::getSaveFileName(QDir::currentDirPath(),
                               KImageIO::pattern(KImageIO::Writing), this, i18n("Save as..."));
  if(!newName.isEmpty())
  {
    KScribbleView* m = (KScribbleView*)pWorkspace->activeWindow();
    if( m )
    {
      KScribbleDoc* doc =	m->getDocument();
  	  QString format=QFileInfo(newName).extension();
  	  format=format.upper();
		  if(!doc->saveDocument(newName,format))
  	  {
  		  KMessageBox::error (this,i18n("Could not save the current document !"), i18n("I/O Error !"));
				return;
			}
      doc->changedViewList();
      setWndTitle(m);
    }
		
  }

  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotFileClose()
{
  slotStatusMsg(i18n("Closing file..."));
	
	KScribbleView* m = (KScribbleView*)pWorkspace->activeWindow();
	if( m )
	{
		KScribbleDoc* doc=m->getDocument();
    doc->closeDocument();
  }

	
  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotFilePrint()
{
  slotStatusMsg(i18n("Printing..."));
	
	KScribbleView* m = (KScribbleView*) pWorkspace->activeWindow();
	if ( m )
		m->print( printer );

  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotFileQuit()
{
  slotStatusMsg(i18n("Exiting..."));
  saveOptions();
  // close the first window, the list makes the next one the first again.
  // This ensures that queryClose() is called on each window to ask for closing
  KTMainWindow* w;
  if(memberList)
  {
    for(w=memberList->first(); w!=0; w=memberList->first())
    {
      // only close the window if the closeEvent is accepted. If the user presses Cancel on the saveModified() dialog,
      // the window and the application stay open.
      if(!w->close())
			break;
    }
  }	
  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotEditUndo()
{
  slotStatusMsg(i18n("Reverting last action..."));
	
	KScribbleView* m = (KScribbleView*) pWorkspace->activeWindow();
	if ( m )
//		m->undo();

  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotEditCut()
{
  slotStatusMsg(i18n("Cutting selection..."));
	
	KScribbleView* m = (KScribbleView*) pWorkspace->activeWindow();
	if ( m )
		m->cutSelection();	

  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotEditCopy()
{
  slotStatusMsg(i18n("Copying selection to clipboard..."));
	
	KScribbleView* m = (KScribbleView*) pWorkspace->activeWindow();
	if ( m )
		m->copySelection();
		
  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotEditPaste()
{
  slotStatusMsg(i18n("Inserting clipboard contents..."));
	
	KScribbleView* m = (KScribbleView*) pWorkspace->activeWindow();
	if ( m )
		m->pasteSelection();
		
  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotEditClearAll()
{
  slotStatusMsg(i18n("Clearing the document contents..."));
	
	KScribbleView* m = (KScribbleView*) pWorkspace->activeWindow();
	if ( m ){
		KScribbleDoc* pDoc = m->getDocument();
		pDoc->editClearAll();
	}
  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotPenBrush()
{
  slotStatusMsg(i18n("Setting brush width..."));

  // get one window with document for a current pen width
  QWidgetList windows = pWorkspace->windowList();
  KScribbleView* m = (KScribbleView*)windows.at(0);
	KScribbleDoc* pDoc = m->getDocument();
  int curr_width=pDoc->penWidth();

  // create the dialog, get the new width and set the pen width for all documents
  KPenBrushDlg* dlg= new KPenBrushDlg(curr_width,this);
  if(dlg->exec()){
    int width=dlg->width();
  	for ( int i = 0; i < int(windows.count()); ++i )
  	{
    	m = (KScribbleView*)windows.at(i);
    	if ( m )
    	{
    		pDoc = m->getDocument();
        pDoc->setPenWidth(width);
    	}
  	}
  }
  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotPenColor()
{
  slotStatusMsg(i18n("Selecting pen color..."));

  QColor myColor;
  int result = KColorDialog::getColor( myColor, this );
  if ( result == KColorDialog::Accepted )
  {
  	QWidgetList windows = pWorkspace->windowList();
  	KScribbleDoc* pDoc;
  	KScribbleView* m;
  	for ( int i = 0; i < int(windows.count()); ++i )
  	{
    	m = (KScribbleView*)windows.at(i);
    	if ( m )
    	{
    		pDoc = m->getDocument();
    	  pDoc->setPenColor(myColor);
    	}
  	}
	}
  slotStatusMsg(i18n("Ready."));
}


void KScribbleApp::slotViewToolBar()
{
  slotStatusMsg(i18n("Toggle the toolbar..."));
  ///////////////////////////////////////////////////////////////////
  // turn Toolbar on or off
  if( menuBar()->isItemChecked(ID_VIEW_TOOLBAR))
  {
    menuBar()->setItemChecked(ID_VIEW_TOOLBAR, false);
    enableToolBar(KToolBar::Hide);
  }
  else
  {
    menuBar()->setItemChecked(ID_VIEW_TOOLBAR, true);
    enableToolBar(KToolBar::Show);
  }		

  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotViewStatusBar()
{
  slotStatusMsg(i18n("Toggle the statusbar..."));
  ///////////////////////////////////////////////////////////////////
  //turn Statusbar on or off
  if( menuBar()->isItemChecked(ID_VIEW_STATUSBAR))
  {
    menuBar()->setItemChecked(ID_VIEW_STATUSBAR, false);
    enableStatusBar(KStatusBar::Hide);
  }
  else
  {
    menuBar()->setItemChecked(ID_VIEW_STATUSBAR, true);
    enableStatusBar(KStatusBar::Show);
  }

  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotWindowNewWindow()
{
  slotStatusMsg(i18n("Opening a new application window..."));
	
	KScribbleView* m = (KScribbleView*) pWorkspace->activeWindow();
	if ( m ){
 		KScribbleDoc* doc = m->getDocument();
  	createClient(doc);
	}

  slotStatusMsg(i18n("Ready."));
}

void KScribbleApp::slotStatusMsg(const QString &text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message permanently
  statusBar()->clear();
  statusBar()->changeItem(text, ID_STATUS_MSG);
}


void KScribbleApp::slotStatusHelpMsg(const QString &text)
{
  ///////////////////////////////////////////////////////////////////
  // change status message of whole statusbar temporary (text, msec)
  statusBar()->message(text, 2000);
}

void KScribbleApp::windowMenuAboutToShow()
{
	pWindowMenu->clear();
	
	pWindowMenu->insertItem(i18n("&New Window"), ID_WINDOW_NEW_WINDOW);
  pWindowMenu->insertItem(i18n("&Cascade"), pWorkspace, SLOT(cascade() ),0 , ID_WINDOW_CASCADE );
  pWindowMenu->insertItem(i18n("&Tile"), pWorkspace, SLOT(tile() ),0 , ID_WINDOW_TILE );
	
	if ( pWorkspace->windowList().isEmpty() ) {
		disableCommand(ID_WINDOW_NEW_WINDOW);
		disableCommand(ID_WINDOW_CASCADE);
		disableCommand(ID_WINDOW_TILE);
	}
	
	pWindowMenu->insertSeparator();
	
	QWidgetList windows = pWorkspace->windowList();
	for ( int i = 0; i < int(windows.count()); ++i ) {
		int id = pWindowMenu->insertItem(QString("&%1 ").arg(i+1)+windows.at(i)->caption(), this, SLOT( windowMenuActivated( int ) ) );
		pWindowMenu->setItemParameter( id, i );
	  pWindowMenu->setItemChecked( id, pWorkspace->activeWindow() == windows.at(i) );
	}
}

void KScribbleApp::windowMenuActivated( int id )
{
  QWidget* w = pWorkspace->windowList().at( id );
  if ( w )
		w->setFocus();
}


void KScribbleApp::commandCallback(int id_)
{
  switch (id_)
  {
    case ID_FILE_NEW:
    	 slotFileNew();
         break;

    case ID_FILE_OPEN:
         slotFileOpen();
         break;

    case ID_FILE_SAVE:
         slotFileSave();
         break;

    case ID_FILE_SAVE_AS:
         slotFileSaveAs();
         break;

    case ID_FILE_CLOSE:
         slotFileClose();
         break;

    case ID_FILE_PRINT:
         slotFilePrint();
         break;

    case ID_FILE_QUIT:
         slotFileQuit();
         break;

    case ID_EDIT_CUT:
         slotEditCut();
         break;

    case ID_EDIT_COPY:
         slotEditCopy();
         break;

    case ID_EDIT_PASTE:
         slotEditPaste();
         break;
  
    case ID_EDIT_CLEAR_ALL:
         slotEditClearAll();
         break;

    case ID_PEN_BRUSH:
         slotPenBrush();
         break;

    case ID_PEN_COLOR:
         slotPenColor();
         break;

    case ID_VIEW_TOOLBAR:
         slotViewToolBar();
         break;

    case ID_VIEW_STATUSBAR:
         slotViewStatusBar();
         break;

    case ID_WINDOW_NEW_WINDOW:
         slotWindowNewWindow();
    	 break;

    default:
         break;
  }
}

void KScribbleApp::statusCallback(int id_)
{
  switch (id_)
  {
    case ID_FILE_NEW:
         slotStatusHelpMsg(i18n("Creates a new document"));
         break;

    case ID_FILE_OPEN:
         slotStatusHelpMsg(i18n("Opens an existing document"));
         break;

    case ID_FILE_OPEN_RECENT:
         slotStatusHelpMsg(i18n("Opens a recently used file"));
         break;

    case ID_FILE_SAVE:
         slotStatusHelpMsg(i18n("Saves the currently active document"));
         break;

    case ID_FILE_SAVE_AS:
         slotStatusHelpMsg(i18n("Saves the currently active document as under a new filename"));
         break;

    case ID_FILE_CLOSE:
         slotStatusHelpMsg(i18n("Closes the currently active document"));
         break;

    case ID_FILE_PRINT:
         slotStatusHelpMsg(i18n("Prints out the actual document"));
         break;

    case ID_FILE_QUIT:
         slotStatusHelpMsg(i18n("Quits the application"));
         break;

    case ID_EDIT_UNDO:
         slotStatusHelpMsg(i18n("Reverts the last editing action"));
         break;

    case ID_EDIT_CUT:
         slotStatusHelpMsg(i18n("Cuts the selected section and puts it to the clipboard"));
         break;

    case ID_EDIT_COPY:
         slotStatusHelpMsg(i18n("Copies the selected section to the clipboard"));
         break;

    case ID_EDIT_PASTE:
         slotStatusHelpMsg(i18n("Pastes the clipboard contents to actual position"));
         break;

    case ID_EDIT_CLEAR_ALL:
         slotStatusHelpMsg(i18n("Clears the document contents"));
         break;

    case ID_PEN_BRUSH:
         slotStatusHelpMsg(i18n("Sets the pen width"));
         break;

    case ID_PEN_COLOR:
         slotStatusHelpMsg(i18n("Sets the current pen color"));
         break;

    case ID_VIEW_TOOLBAR:
         slotStatusHelpMsg(i18n("Enables/disables the toolbar"));
         break;

    case ID_VIEW_STATUSBAR:
         slotStatusHelpMsg(i18n("Enables/disables the statusbar"));
         break;

    case ID_WINDOW_NEW_WINDOW:
         slotStatusHelpMsg(i18n("Opens a new view for the current document"));
         break;

    case ID_WINDOW_CASCADE:
         slotStatusHelpMsg(i18n("Cascades all windows"));
         break;

    case ID_WINDOW_TILE:
         slotStatusHelpMsg(i18n("Tiles all windows"));
         break;

    default:
         break;
  }
}
/** accepts drops and opens a new document
for each drop */
void KScribbleApp::dropEvent( QDropEvent* e){

  QImage img;
  if ( QImageDrag::decode(e, img) )
  {
    KScribbleDoc* doc = new KScribbleDoc();
		untitledCount+=1;
		QString fileName=QString(i18n("Untitled%1")).arg(untitledCount);
		doc->setPathName(fileName);
		doc->setTitle(fileName);
    doc->newDocument();
    pDocList->append(doc);
    KPixmap tmp;
    tmp.resize(img.size());
    tmp.convertFromImage(img);
    doc->setPixmap(tmp);
    doc->resizeDocument(tmp.size());
    doc->setModified();
    createClient(doc);
  }
}
/** accepts drag events for images */
void KScribbleApp::dragEnterEvent( QDragEnterEvent* e){
  e->accept(QImageDrag::canDecode(e));
}
